/*
 * #%L
 * Maven License Plugin
 * 
 * $Id: AbstractLicenseWithDescriptorMojo.java 1857 2010-10-17 01:09:17Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-license-plugin/tags/maven-license-plugin-2.4.1/src/main/java/org/nuiton/license/plugin/AbstractLicenseWithDescriptorMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.license.plugin;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.nuiton.license.plugin.model.License;
import org.nuiton.license.plugin.model.LicenseStore;
import org.nuiton.license.plugin.model.descriptor.LicenseProjectDescriptor;
import org.nuiton.license.plugin.model.descriptor.io.xpp3.LicenseProjectDescriptorXpp3Reader;
import org.nuiton.plugin.PluginWithEncoding;

import java.io.File;
import java.io.FileReader;
import java.util.Arrays;

/**
 * Abstract mojo which using {@link #descriptor} file and owns a
 * {@link #licenseStore}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.1
 */
public abstract class AbstractLicenseWithDescriptorMojo extends AbstractLicenseMojo implements PluginWithEncoding {

    /**
     * Encoding used to read and writes files.
     * <p/>
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     *
     * @parameter expression="${license.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 2.1
     */
    private String encoding;

    /**
     * The project license descriptor file.
     * <p/>
     * <b>Note:</b> This parameter is no more required as it will be removed in
     * version 3.
     *
     * @parameter expression="${license.descriptor}" default-value="src/license/project.xml"
     * @since 2.1
     * @deprecated since 2.4.1, in version 3 we will no more use descriptor
     */
    @Deprecated
    private File descriptor;

    /**
     * To specify an external extra licenses repository resolver (says the base
     * url of the repository where the {@code license.properties} is present).
     *
     * @parameter expression="${license.licenseResolver}"
     * @since 2.1
     */
    private String licenseResolver;

    /**
     * A flag to keep a backup of every modified file.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 2.1
     */
    private boolean keepBackup;

    /**
     * Name of the license to use in the project.
     *
     * @parameter expression="${license.licenseName}"
     * @since 2.4
     */
    private String licenseName;

    /** store of licenses */
    private LicenseStore licenseStore;

    /**
     * descriptor of project.
     *
     * @deprecated since 2.4.1, in version 3 we will no more use a descriptor file to describe project license.
     */
    @Deprecated
    private LicenseProjectDescriptor licenseProjectDescriptor;

    /**
     * When is sets to {@code true}, will skip execution.
     * <p/>
     * This will take effects in method {@link #checkSkip()}.
     * So the method {@link #doAction()} will never be invoked.
     *
     * @return {@code true} if goal will not be executed
     */
    public abstract boolean isSkip();

    /**
     * Changes internal state {@code skip} to execute (or not) goal.
     *
     * @param skip new state value
     */
    public abstract void setSkip(boolean skip);

    @Override
    protected boolean checkSkip() {
        if (isSkip()) {
            getLog().info("skip flag is on, will skip goal.");
            return false;
        }
        return super.checkSkip();
    }

    @Override
    protected void init() throws Exception {

        if (isSkip()) {
            return;
        }

        // init licenses store
        LicenseStore licenseStore = createLicenseStore(getLicenseResolver());
        setLicenseStore(licenseStore);

        // load project descriptor
        File descriptorFile = new File(getDescriptor().getAbsolutePath());
        if (!descriptorFile.exists()) {

            // try to find the file in the direct parent module
            MavenProject mavenProject = getProject().getParent();
            if (mavenProject != null) {

                File basedir = getProject().getBasedir();

                // try with the parent
                String path = descriptorFile.getAbsolutePath().substring(
                        basedir.getAbsolutePath().length() + 1);
                setDescriptor(new File(mavenProject.getBasedir() +
                                       File.separator + path));
                if (isVerbose()) {
                    getLog().info("try in parent module " + getDescriptor());
                }

            }
            descriptorFile = new File(getDescriptor().getAbsolutePath());
        }

        if (!descriptorFile.exists()) {

            // No descriptor File: use no descriptor

            // check licenseName exists
            checkLicense(licenseName);

            return;
        }

        // Will use the deprecated license project descriptor

        getLog().warn("\n");
        getLog().warn("-----------------------------------------------------------------------------------");
        getLog().warn("The usage of License descriptor file is deprecated and will be removed in version 3");
        getLog().warn("-----------------------------------------------------------------------------------");
        getLog().warn("\n");

        getLog().info("Loading descriptor " + descriptorFile);
        LicenseProjectDescriptor licenseProject = null;
        FileReader reader = new FileReader(descriptorFile);
        try {
            licenseProject =
                    new LicenseProjectDescriptorXpp3Reader().read(reader);
            setLicenseProjectDescriptor(licenseProject);

        } finally {
            reader.close();
        }

        // check that license project is sane (known licenses + header types)
        // + populate default license name in LicenseSet (with main License) if
        // none given for licenseSet
        validateLicenseProjectDescriptor(licenseProject);
    }

    @Deprecated
    protected void validateLicenseProjectDescriptor(LicenseProjectDescriptor licenseProjectDescriptor) throws MojoFailureException {
        String mainLicense = licenseProjectDescriptor.getMainLicense();

        checkLicense(mainLicense);

        setLicenseName(mainLicense);
    }

    public License getMainLicense()
            throws IllegalArgumentException, IllegalStateException, MojoFailureException {
//        LicenseProjectDescriptor licenseProject = getLicenseProjectDescriptor();
//        if (licenseProject != null) {
//
//            // use the main license from the license project descriptor
//            licenseName = licenseProject.getMainLicense();
//        }

        // check license exists
        checkLicense(licenseName);

        // obtain license from his name
        License mainLicense = getLicense(licenseName);
        return mainLicense;
    }

    public License getLicense(String licenseName)
            throws IllegalArgumentException, IllegalStateException {
        if (StringUtils.isEmpty(licenseName)) {
            throw new IllegalArgumentException(
                    "licenseName can not be null, nor empty");
        }
        LicenseStore licenseStore = getLicenseStore();
        if (licenseStore == null) {
            throw new IllegalStateException("No license store initialized!");
        }
        License mainLicense = licenseStore.getLicense(licenseName);
        return mainLicense;
    }

    /**
     * Check if the given license name is valid (not null, nor empty) and
     * exists in the license store.
     *
     * @param licenseName the name of the license to check
     * @throws IllegalArgumentException if license is not valid
     * @throws IllegalStateException    if license store is not initialized
     * @throws MojoFailureException     if license does not exist
     * @since 2.4.1
     */
    protected void checkLicense(String licenseName)
            throws IllegalArgumentException,
            IllegalStateException, MojoFailureException {
        if (StringUtils.isEmpty(licenseName)) {
            throw new IllegalArgumentException(
                    "licenseName can not be null, nor empty");
        }
        LicenseStore licenseStore = getLicenseStore();
        if (licenseStore == null) {
            throw new IllegalStateException("No license store initialized!");
        }
        License mainLicense = licenseStore.getLicense(licenseName);
        if (mainLicense == null) {
            throw new MojoFailureException(
                    "License named '" + mainLicense +
                    "' is unknown, use one of " +
                    Arrays.toString(licenseStore.getLicenseNames()));
        }
    }

    @Override
    public final String getEncoding() {
        return encoding;
    }

    @Override
    public final void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    @Deprecated
    public File getDescriptor() {
        return descriptor;
    }

    public boolean isKeepBackup() {
        return keepBackup;
    }

    public String getLicenseName() {
        return licenseName;
    }

    public String getLicenseResolver() {
        return licenseResolver;
    }

    public LicenseStore getLicenseStore() {
        return licenseStore;
    }

    @Deprecated
    public LicenseProjectDescriptor getLicenseProjectDescriptor() {
        return licenseProjectDescriptor;
    }

    public void setKeepBackup(boolean keepBackup) {
        this.keepBackup = keepBackup;
    }

    @Deprecated
    public void setDescriptor(File descriptor) {
        this.descriptor = descriptor;
    }

    public void setLicenseResolver(String licenseResolver) {
        this.licenseResolver = licenseResolver;
    }

    public void setLicenseName(String licenseName) {
        this.licenseName = licenseName;
    }

    public void setLicenseStore(LicenseStore licenseStore) {
        this.licenseStore = licenseStore;
    }

    @Deprecated
    public void setLicenseProjectDescriptor(
            LicenseProjectDescriptor licenseProjectDescriptor) {
        this.licenseProjectDescriptor = licenseProjectDescriptor;
    }
}
