/*
 * #%L
 * Maven License Plugin
 * 
 * $Id: UpdateFileHeaderMojo.java 1759 2010-04-16 17:17:45Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-license-plugin/tags/maven-license-plugin-2.3/src/main/java/org/nuiton/license/plugin/UpdateFileHeaderMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.license.plugin;

import org.apache.commons.lang.StringUtils;
import org.nuiton.license.plugin.header.*;
import org.nuiton.license.plugin.header.transformer.FileHeaderTransformer;
import org.nuiton.license.plugin.model.License;
import org.nuiton.license.plugin.model.descriptor.FileSet;
import org.nuiton.license.plugin.model.descriptor.Header;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.util.*;

/**
 * The goal to update (or add) the header on some files described in
 * {@link #descriptor} file.
 * <p/>
 * This goal replace the {@code update-header} goal which can not deal with
 * Copyright.
 * <p/>
 * This goal use a specific project file descriptor {@code project.xml} to
 * describe all files to update for a whole project.
 *
 * @author tchemit <chemit@codelutin.com>
 * @requiresProject true
 * @goal update-file-header
 * @since 2.1
 */
public class UpdateFileHeaderMojo extends AbstractLicenseWithDescriptorMojo implements FileHeaderProcessorConfiguration {

    /**
     * Name of project (or module).
     * <p/>
     * Will be used as description section of new header.
     *
     * @parameter expression="${license.projectName}" default-value="${project.name}"
     * @required
     * @since 2.1
     */
    protected String projectName;

    /**
     * Name of project's organization.
     * <p/>
     * Will be used as copyrigth's holder in new header.
     *
     * @parameter expression="${license.organizationName}" default-value="${project.organization.name}"
     * @required
     * @since 2.1
     */
    protected String organizationName;

    /**
     * Inception year of the project.
     * <p/>
     * Will be used as first year of copyright section in new header.
     *
     * @parameter expression="${license.inceptionYear}" default-value="${project.inceptionYear}"
     * @required
     * @since 2.1
     */
    protected String inceptionYear;

    /**
     * A flag to add svn:keywords on new header.
     * <p/>
     * Will add svn keywords :
     * <pre>Author, Id, Rev, URL and Date</pre>
     *
     * @parameter expression="${license.addSvnKeyWords}" default-value="false"
     * @since 2.1
     */
    protected boolean addSvnKeyWords;

    /**
     * A flag to update copyright application time (change copyright last year
     * if required) according to the last commit made on the processed file.
     *
     * @parameter expression="${license.updateCopyright}" default-value="false"
     * @since 2.1
     */
    protected boolean updateCopyright;

    /**
     * A tag to place on files that will be ignored by the plugin.
     * <p/>
     * Sometimes, it is necessary to do this when file is under a specific license.
     * <p/>
     * <b>Note:</b> If no sets, will use the default tag {@code %}%Ignore-License
     *
     * @parameter expression="${license.ignoreTag}"
     * @since 2.1
     */
    protected String ignoreTag;

    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${license.skipUpdateLicense}" default-value="false"
     * @since 2.1
     */
    protected boolean skipUpdateLicense;

    /**
     * A flag to test plugin but modify no file.
     *
     * @parameter expression="${dryRun}" default-value="false"
     * @since 2.1
     */
    protected boolean dryRun;

    /**
     * A flag to clear everything after execution.
     * <p/>
     * <b>Note:</b> This property should ONLY be used for test purpose.
     *
     * @parameter expression="${license.clearAfterOperation}" default-value="true"
     * @since 2.1
     */
    protected boolean clearAfterOperation;

    /**
     * @component role="org.nuiton.processor.Processor" roleHint="file-header"
     * @since 2.1
     */
    protected FileHeaderProcessor processor;

    /**
     * @component role="org.nuiton.license.plugin.header.FileHeaderFilter" roleHint="update-file-header"
     * @since 2.1
     */
    protected UpdateFileHeaderFilter filter;

    /** internal file header transformer */
    protected FileHeaderTransformer transformer;

    /** internal default file header */
    protected FileHeader header;

    /** timestamp used for generation */
    protected long timestamp;

    /**
     * Defines state of a file after process.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.1
     */
    enum FileState {

        /** file was updated */
        update,

        /** file was up to date */
        uptodate,

        /** something was added on file */
        add,

        /** file was ignored */
        ignore,

        /** treatment failed for file */
        fail;

        /**
         * register a file for this state on result dictionary.
         *
         * @param file   file to add
         * @param result dictionary to update
         */
        public void addFile(File file,
                            EnumMap<FileState, Set<File>> result) {
            Set<File> fileSet = result.get(this);
            if (fileSet == null) {
                fileSet = new HashSet<File>();
                result.put(this, fileSet);
            }
            fileSet.add(file);
        }
    }

    /** set of processed files */
    protected Set<File> processedFiles;

    /** by file state files treated */
    protected EnumMap<FileState, Set<File>> result;

    @Override
    public void init() throws Exception {

        if (isSkip()) {
            return;
        }

        if (StringUtils.isEmpty(getIgnoreTag())) {

            // use default value
            setIgnoreTag("%" + "%Ignore-License");
        }

        if (isVerbose()) {

            // print availables comment styles (transformers)
            StringBuilder buffer = new StringBuilder();
            buffer.append("config - available comment styles :");
            String commentFormat = "\n  * %1$s (%2$s)";
            for (String transformerName : getTransformers().keySet()) {
                FileHeaderTransformer transformer =
                        getTransformer(transformerName);
                String str = String.format(commentFormat,
                                           transformer.getName(),
                                           transformer.getDescription()
                );
                buffer.append(str);
            }
            getLog().info(buffer.toString());
        }

        if (isUpdateCopyright()) {

            getLog().warn("\n\nupdateCopyright is not still available...\n\n");
            //TODO-TC20100409 checks scm
            // checks scm is ok
            // for the moment, will only deal with svn except if scm
            // offers a nice api to obtain last commit date on a file

        }

        // set timestamp used for temporary files
        setTimestamp(System.nanoTime());

        getFilter().setUpdateCopyright(isUpdateCopyright());
        getFilter().setLog(getLog());
        getProcessor().setConfiguration(this);
        getProcessor().setFilter(filter);

        super.init();
    }

    @Override
    public void doAction() throws Exception {

        long t0 = System.nanoTime();

        clear();

        processedFiles = new HashSet<File>();
        result = new EnumMap<FileState, Set<File>>(FileState.class);

        try {

            for (Header header : getLicenseProjectDescriptor().getHeaders()) {

                processHeader(header);
            }

        } finally {

            int nbFiles = getProcessedFiles().size();
            if (nbFiles == 0) {
                getLog().warn("No file to scan.");
            } else {
                String delay = PluginHelper.convertTime(System.nanoTime() - t0);
                String message = String.format(
                        "Scan %s file%s header done in %s.",
                        nbFiles,
                        nbFiles > 1 ? "s" : "",
                        delay
                );
                getLog().info(message);
            }
            Set<FileState> states = result.keySet();
            if (states.size() == 1 && states.contains(FileState.uptodate)) {
                // all files where up to date
                getLog().info("All files are up-to-date.");
            } else {

                StringBuilder buffer = new StringBuilder();
                for (FileState state : FileState.values()) {

                    reportType(state, buffer);
                }

                getLog().info(buffer.toString());
            }

            // clean internal states
            if (isClearAfterOperation()) {
                clear();
            }
        }
    }

    protected void processHeader(Header header) throws IOException {

        // obtain license from definition
        String licenseName = header.getLicenseName();
        License license = getLicense(licenseName);

        getLog().info("Process header '" + header.getCommentStyle() + "'");
        getLog().info(" - using " + license.getDescription());

        // use header transformer according to comment style given in header
        setTransformer(getTransformer(header.getCommentStyle()));

        // file header to use if no header is found on a file
        FileHeader defaultFileHeader = buildDefaultFileHeader(
                license,
                getProjectName(),
                getInceptionYear(),
                getOrganizationName(),
                isAddSvnKeyWords(),
                getEncoding()
        );

        // change default license header in processor
        setHeader(defaultFileHeader);

        // update processor filter
        getProcessor().populateFilter();

        for (FileSet fileSet : header.getFileSets()) {

            File basedir = new File(getProject().getBasedir(),
                                    fileSet.getBasedir());
            if (getLog().isDebugEnabled()) {
                getLog().debug(" - process file set with basedir : " + basedir);
            }

            List<String> includes = fileSet.getIncludes();
            if (includes.isEmpty()) {

                // it means include all
                includes.add("**/*");
            }
            List<String> excludes = fileSet.getExcludes();

            Map<File, String[]> filestoTreate = new TreeMap<File, String[]>();

            // obtain files to treate
            getFilesToTreateForRoots(
                    includes.toArray(new String[includes.size()]),
                    excludes.isEmpty() ? null :
                    excludes.toArray(new String[excludes.size()]),
                    Arrays.asList(basedir.getAbsolutePath()),
                    filestoTreate,
                    null
            );

            try {
                for (Map.Entry<File, String[]> entry :
                        filestoTreate.entrySet()) {

                    // treate all files of entry
                    processFileEntry(entry.getKey(), entry.getValue());
                }
            } finally {
                filestoTreate.clear();
            }
        }
    }

    protected void processFileEntry(File entryBasedir,
                                    String[] paths) throws IOException {

        getLog().info(" - " + paths.length + " file(s) to treate in " + entryBasedir);
        for (String path : paths) {
            File file = new File(entryBasedir, path);
            if (getProcessedFiles().contains(file)) {
                getLog().info(" - skip already processed file " + file);
                continue;
            }

            // output file
            File processFile =
                    new File(file.getAbsolutePath() + "_" + getTimestamp());
            boolean doFinalize = false;
            try {
                doFinalize = processFile(file, processFile);
            } catch (Exception e) {
                getLog().warn("skip failed file : " +
                              e.getMessage() +
                              (e.getCause() == null ? "" :
                               " Cause : " + e.getCause().getMessage()), e
                );
                FileState.fail.addFile(file, getResult());
                doFinalize = false;
            } finally {

                // always clean processor internal states
                getProcessor().reset();

                // whatever was the result, this file is treated.
                getProcessedFiles().add(file);

                if (doFinalize) {
                    finalizeFile(file, processFile);
                } else {
                    deleteFile(processFile);
                }
            }
        }
    }

    /**
     * Process the given {@code file} and save the result in the given
     * {@code processFile}.
     *
     * @param file        the file to process
     * @param processFile the ouput processed file
     * @return {@code true} if processFile can be finalize, otherwise need to be delete
     * @throws IOException if any pb while treatment
     */
    protected boolean processFile(File file,
                                  File processFile) throws IOException {


        if (getLog().isDebugEnabled()) {
            getLog().debug(" - process file " + file);
            getLog().debug(" - will process into file " + processFile);
        }

        String content;

        try {

            // check before all that file should not be skip by the ignoreTag
            // this is a costy operation
            //TODO-TC-20100411 We should process always from the read content not reading again from file

            content = PluginHelper.readAsString(file, getEncoding());

        } catch (IOException e) {
            throw new IOException("Could not obtain content of file " + file);
        }

        //check that file is not marked to be ignored
        if (content.contains(getIgnoreTag())) {
            getLog().info(
                    " - ignore file (detected " + getIgnoreTag() + ") " + file);

            FileState.ignore.addFile(file, getResult());

            return false;
        }

        FileHeaderProcessor processor = getProcessor();

        // process file to detect header

        try {
            processor.process(file, processFile);
        } catch (IllegalStateException e) {
            // could not obtain existing header
            throw new InvalideFileHeaderException(
                    "Could not extract header on file " + file, e);
        } catch (Exception e) {
            if (e instanceof InvalideFileHeaderException) {
                throw (InvalideFileHeaderException) e;
            }
            throw new IOException("Could not process file " + file, e);
        }


        if (processor.isTouched()) {

            if (isVerbose()) {
                getLog().info(" - header was updated for " + file);
            }
            if (processor.isModified()) {

                // header content has changed
                // must copy back process file to file (if not dry run)

                FileState.update.addFile(file, getResult());
                return true;

            }

            FileState.uptodate.addFile(file, getResult());
            return false;
        }

        // header was not fully (or not at all) detected in file

        if (processor.isDetectHeader()) {

            // file has not a valid header (found a start process atg, but
            // not an ending one), can not do anything
            throw new InvalideFileHeaderException(
                    "Could not find header end on file " + file);
        }

        // no header at all, add a new header

        getLog().info(" - adding license header on file " + file);

        //FIXME-TC-20100409 form xml files must add header after a xml prolog line
        content = getTransformer().addHeader(
                getFilter().getFullHeaderContent(),
                content
        );

        if (!isDryRun()) {
            writeFile(processFile, content, getEncoding());
        }

        FileState.add.addFile(file, getResult());
        return true;
    }

    protected void finalizeFile(File file, File processFile) throws IOException {

        if (isKeepBackup() && !isDryRun()) {
            File backupFile = getBackupFile(file);

            if (backupFile.exists()) {

                // always delete backup file, before the renaming
                deleteFile(backupFile);
            }

            if (isVerbose()) {
                getLog().debug(" - backup original file " + file);
            }

            renameFile(file, backupFile);
        }

        if (isDryRun()) {

            // dry run, delete temporary file
            deleteFile(processFile);
        } else {

            // replace file with
            renameFile(processFile, file);
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        clear();
    }

    protected void clear() {
        Set<File> files = getProcessedFiles();
        if (files != null) {
            files.clear();
        }
        EnumMap<FileState, Set<File>> result = getResult();
        if (result != null) {
            for (Set<File> fileSet : result.values()) {
                fileSet.clear();
            }
            result.clear();
        }
    }

    protected void reportType(FileState state, StringBuilder buffer) {
        String operation = state.name();

        Set<File> set = getFiles(state);
        if (set == null || set.isEmpty()) {
            if (isVerbose()) {
                buffer.append("\n * no header to ");
                buffer.append(operation);
                buffer.append(".");
            }
            return;
        }
        buffer.append("\n * ").append(operation).append(" header on ");
        buffer.append(set.size());
        if (set.size() == 1) {
            buffer.append(" file.");
        } else {
            buffer.append(" files.");
        }
        if (isVerbose()) {
            for (File file : set) {
                buffer.append("\n   - ").append(file);
            }
        }
    }

    public boolean isClearAfterOperation() {
        return clearAfterOperation;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public String getProjectName() {
        return projectName;
    }

    public String getInceptionYear() {
        return inceptionYear;
    }

    public String getOrganizationName() {
        return organizationName;
    }

    public boolean isUpdateCopyright() {
        return updateCopyright;
    }

    public String getIgnoreTag() {
        return ignoreTag;
    }

    public boolean isDryRun() {
        return dryRun;
    }

    public UpdateFileHeaderFilter getFilter() {
        return filter;
    }

    @Override
    public FileHeader getFileHeader() {
        return header;
    }

    @Override
    public FileHeaderTransformer getTransformer() {
        return transformer;
    }

    @Override
    public boolean isSkip() {
        return skipUpdateLicense;
    }

    public Set<File> getProcessedFiles() {
        return processedFiles;
    }

    public EnumMap<FileState, Set<File>> getResult() {
        return result;
    }

    public Set<File> getFiles(FileState state) {
        return result.get(state);
    }

    public boolean isAddSvnKeyWords() {
        return addSvnKeyWords;
    }

    public FileHeaderProcessor getProcessor() {
        return processor;
    }

    @Override
    public void setSkip(boolean skipUpdateLicense) {
        this.skipUpdateLicense = skipUpdateLicense;
    }

    public void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public void setProjectName(String projectName) {
        this.projectName = projectName;
    }

    public void setSkipUpdateLicense(boolean skipUpdateLicense) {
        this.skipUpdateLicense = skipUpdateLicense;
    }

    public void setInceptionYear(String inceptionYear) {
        this.inceptionYear = inceptionYear;
    }

    public void setOrganizationName(String organizationName) {
        this.organizationName = organizationName;
    }

    public void setUpdateCopyright(boolean updateCopyright) {
        this.updateCopyright = updateCopyright;
    }

    public void setIgnoreTag(String ignoreTag) {
        this.ignoreTag = ignoreTag;
    }

    public void setAddSvnKeyWords(boolean addSvnKeyWords) {
        this.addSvnKeyWords = addSvnKeyWords;
    }

    public void setClearAfterOperation(boolean clearAfterOperation) {
        this.clearAfterOperation = clearAfterOperation;
    }

    public void setTransformer(FileHeaderTransformer transformer) {
        this.transformer = transformer;
    }

    public void setHeader(FileHeader header) {
        this.header = header;
    }

    public void setProcessor(FileHeaderProcessor processor) {
        this.processor = processor;
    }

    public void setFilter(UpdateFileHeaderFilter filter) {
        this.filter = filter;
    }


}
