/*
 * #%L
 * Maven License Plugin
 * 
 * $Id: AddThirdPartyMojo.java 1755 2010-04-16 11:56:15Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-license-plugin/tags/maven-license-plugin-2.2/src/main/java/org/nuiton/license/plugin/AddThirdPartyMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.license.plugin;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactCollector;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.model.License;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.shared.dependency.tree.DependencyNode;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilder;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilderException;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.util.*;

/**
 * Le goal pour copier le fichier THIRD-PARTY.txt (contenant les licenses de
 * toutes les dependances du projet) dans le classpath (et le generer s'il
 * n'existe pas).
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal add-third-party
 * @phase generate-resources
 * @requiresDependencyResolution test
 * @requiresProject true
 * @since 2.2 (was previously {@code AddThirdPartyFileMojo}).
 */
public class AddThirdPartyMojo extends AbstractLicenseMojo {

    private static final String unknownLicenseMessage = "Unknown license";

    /**
     * Repertoire de sortie des classes (classpath).
     *
     * @parameter expression="${license.outputDirectory}" default-value="target/generated-sources/license"
     * @required
     * @since 1.0.0
     */
    protected File outputDirectory;

    /**
     * Fichier ou ecrire les licences des dependances.
     *
     * @parameter expression="${license.thirdPartyFilename}"
     * default-value="THIRD-PARTY.txt"
     * @required
     * @since 1.0.0
     */
    protected String thirdPartyFilename;

    /**
     * Un flag pour faire une copie nommé dans META-INF (prefixe avec le nom de
     * l'artifact).
     *
     * @parameter expression="${license.generateBundle}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean generateBundle;

    /**
     * The path of the bundled third party file to produce when
     * {@link #generateBundle} is on.
     * <p/>
     * <b>Note:</b> This option is not available for {@code pom} module types.
     *
     * @parameter expression="${license.bundleThirdPartyPath}"  default-value="META-INF/${project.artifactId}-THIRD-PARTY.txt"
     * @since 2.2
     */
    protected String bundleThirdPartyPath;

    /**
     * Un flag pour forcer la generation.
     *
     * @parameter expression="${license.force}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean force;

    /**
     * Un flag pour conserver un backup des fichiers modifies.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean keepBackup;

    /**
     * Un flag pour faire une copie nommé dans META-INF (prefixe avec le nom de
     * l'artifact).
     *
     * @parameter expression="${license.generateBundle}"  default-value="false"
     * @since 1.0.0
     * @deprecated since 2.2 (use instead {@code generateBundle}).
     */
    @Deprecated
    protected boolean copyToMETA_INF;

    /**
     * Local Repository.
     *
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected ArtifactRepository localRepository;

    /**
     * Remote repositories used for the project.
     *
     * @parameter expression="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected List<?> remoteRepositories;

    /**
     * Dependency tree builder component.
     *
     * @component
     */
    protected DependencyTreeBuilder dependencyTreeBuilder;

    /**
     * Artifact Factory component.
     *
     * @component
     */
    protected ArtifactFactory factory;

    /**
     * Artifact metadata source component.
     *
     * @component
     */
    protected ArtifactMetadataSource artifactMetadataSource;

    /**
     * Artifact collector component.
     *
     * @component
     */
    protected ArtifactCollector collector;

    /**
     * Maven Project Builder component.
     *
     * @component
     */
    protected MavenProjectBuilder mavenProjectBuilder;

    /**
     * content of third party file (only computed if {@link #force} is active or
     * the {@link #thirdPartyFile} does not exist, or is not up-to-date.
     */
    protected String thirdPartyFileContent;

    protected File thirdPartyFile;

    boolean doGenerate;

    public static final String NO_DEPENDENCIES_MESSAGE = "the project has no dependencies.";

    @Override
    protected boolean checkPackaging() {
        return rejectPackaging(Packaging.pom);
    }

    @Override
    protected void init() throws Exception {
        if (getLog().isDebugEnabled()) {

            // always be verbose in debug mode
            setVerbose(true);
        }

        if (copyToMETA_INF) {
            getLog().warn("\\n copyToMETA_INF is deprecated, prefer" +
                          " use the generateBundle parameter\n\n");
            setGenerateBundle(true);
        }

        File file = new File(getOutputDirectory(), getThirdPartyFilename());

        setThirdPartyFile(file);

        setDoGenerate(isForce() || !file.exists() || !isFileNewerThanPomFile(file));

        String content;
        if (isDoGenerate()) {

            if (isVerbose()) {
                getLog().info("will generate third-party content...");
            }

            // build thirdPartyFileContent
            content = buildThirdPartyFilecontent();

        } else {
            // read it from existing third party file
            content = PluginHelper.readAsString(getThirdPartyFile(), getEncoding());
        }

        setThirdPartyFileContent(content);
    }

    @Override
    protected void doAction() throws Exception {

        File target = getThirdPartyFile();
        if (isDoGenerate()) {
            if (isVerbose()) {
                getLog().info("writing third-party file : " + target);
            }
            if (isKeepBackup() && target.exists()) {
                if (isVerbose()) {
                    getLog().info("backup " + target);
                }
                backupFile(target);
            }
            writeFile(target, getThirdPartyFileContent(), getEncoding());
        }
        File output = getOutputDirectory();
        if (isGenerateBundle()) {

            // creates the bundled license file
            File bundleTarget =
                    PluginHelper.getFile(output, getBundleThirdPartyPath());
            copyFile(target, bundleTarget);
        }
        addResourceDir(output, "**/*.txt");
    }

    protected String buildThirdPartyFilecontent() throws DependencyTreeBuilderException {
        DependencyNode dependencyTreeNode;
        ArtifactFilter artifactFilter =
                new ScopeArtifactFilter(Artifact.SCOPE_TEST);
        dependencyTreeNode = dependencyTreeBuilder.buildDependencyTree(
                getProject(),
                localRepository,
                factory,
                artifactMetadataSource,
                artifactFilter,
                collector
        );

        LicenseMap licenseMap = new LicenseMap();

        for (Object o : dependencyTreeNode.getChildren()) {

            buildLicenseMap((DependencyNode) o, licenseMap);
        }

        // log dependencies with no license
        Set<String> unsafeDependencies =
                licenseMap.get(getUnknownLicenseMessage());

        if (unsafeDependencies != null) {
            for (String dep : unsafeDependencies) {
                // no license found for the dependency
                getLog().warn("no license found for dependency " + dep);
            }
        }
        StringBuilder sb = new StringBuilder();
        if (licenseMap.isEmpty()) {
            sb.append(NO_DEPENDENCIES_MESSAGE);
        } else {
            sb.append("List of third-party dependencies grouped by " +
                      "their license type.");
            for (String licenseName : licenseMap.keySet()) {
                sb.append("\n\n").append(licenseName).append(" : ");

                SortedSet<String> projects = licenseMap.get(licenseName);

                for (String projectName : projects) {
                    sb.append("\n  * ").append(projectName);
                }
            }
        }
        String content = sb.toString();
        if (getLog().isDebugEnabled()) {
            getLog().debug("third-party file content :\n" + content);
        }
        return content;
    }

    protected void buildLicenseMap(DependencyNode node, LicenseMap licenseMap) {
        if (node.getState() != DependencyNode.INCLUDED) {
            // this dependency is not included, so do not treate it
            if (isVerbose()) {
                getLog().info("do not include this dependency " +
                              node.toNodeString());
            }
            return;
        }
        Artifact artifact = node.getArtifact();

        if (isVerbose() && getLog().isDebugEnabled()) {
            getLog().debug("treate node " + node.toNodeString());
        }

        if (!Artifact.SCOPE_SYSTEM.equals(artifact.getScope())) {
            try {
                MavenProject artifactProject =
                        getMavenProjectFromRepository(artifact);
                String artifactName = getArtifactName(artifactProject);

                List<?> licenses = artifactProject.getLicenses();

                if (licenses.isEmpty()) {
                    // no license found for the dependency
                    licenseMap.put(getUnknownLicenseMessage(), artifactName);

                } else {
                    for (Object o : licenses) {
                        if (o == null) {
                            getLog().warn("could not acquire the license for "
                                          + artifactName);
                            continue;
                        }
                        License license = (License) o;
                        String licenseKey = license.getName();
                        if (license.getName() == null) {
                            licenseKey = license.getUrl();
                        }
                        licenseMap.put(licenseKey, artifactName);
                    }
                }
            } catch (ProjectBuildingException e) {
                getLog().error("ProjectBuildingException error : ", e);
            }
        }
        if (!node.getChildren().isEmpty()) {
            for (Object o : node.getChildren()) {
                buildLicenseMap((DependencyNode) o, licenseMap);
            }
        }
    }

    protected String getArtifactName(MavenProject artifactProject) {
        StringBuilder sb = new StringBuilder();

        sb.append(artifactProject.getName());
        sb.append(" (");
        sb.append(artifactProject.getGroupId());
        sb.append(":");
        sb.append(artifactProject.getArtifactId());
        sb.append(":");
        sb.append(artifactProject.getVersion());
        sb.append(" - ");
        String url = artifactProject.getUrl();
        sb.append(url == null ? "no url defined" : url);
        sb.append(")");

        return sb.toString();
    }

    /**
     * Get the <code>Maven project</code> from the repository depending the
     * <code>Artifact</code> given.
     *
     * @param artifact an artifact
     * @return the Maven project for the given artifact
     * @throws ProjectBuildingException if any
     */
    protected MavenProject getMavenProjectFromRepository(Artifact artifact)
            throws ProjectBuildingException {

        boolean allowStubModel = false;

        if (!"pom".equals(artifact.getType())) {
            artifact = factory.createProjectArtifact(
                    artifact.getGroupId(),
                    artifact.getArtifactId(),
                    artifact.getVersion(),
                    artifact.getScope()
            );
            allowStubModel = true;
        }

        // TODO: we should use the MavenMetadataSource instead
        return mavenProjectBuilder.buildFromRepository(
                artifact,
                remoteRepositories,
                localRepository,
                allowStubModel
        );
    }

    public static String getUnknownLicenseMessage() {
        return unknownLicenseMessage;
    }

    public String getThirdPartyFilename() {
        return thirdPartyFilename;
    }

    public File getOutputDirectory() {
        return outputDirectory;
    }

    public boolean isForce() {
        return force;
    }

    public boolean isKeepBackup() {
        return keepBackup;
    }

    public boolean isGenerateBundle() {
        return generateBundle;
    }

    public boolean isDoGenerate() {
        return doGenerate;
    }

    public String getBundleThirdPartyPath() {
        return bundleThirdPartyPath;
    }

    public File getThirdPartyFile() {
        return thirdPartyFile;
    }

    public String getThirdPartyFileContent() {
        return thirdPartyFileContent;
    }

    public void setThirdPartyFilename(String thirdPartyFilename) {
        this.thirdPartyFilename = thirdPartyFilename;
    }

    public void setForce(boolean force) {
        this.force = force;
    }

    public void setOutputDirectory(File outputDirectory) {
        this.outputDirectory = outputDirectory;
    }

    public void setKeepBackup(boolean keepBackup) {
        this.keepBackup = keepBackup;
    }

    public void setGenerateBundle(boolean generateBundle) {
        this.generateBundle = generateBundle;
    }

    public void setBundleThirdPartyPath(String bundleThirdPartyPath) {
        this.bundleThirdPartyPath = bundleThirdPartyPath;
    }

    public void setThirdPartyFileContent(String thirdPartyFileContent) {
        this.thirdPartyFileContent = thirdPartyFileContent;
    }

    public void setDoGenerate(boolean doGenerate) {
        this.doGenerate = doGenerate;
    }

    public void setThirdPartyFile(File thirdPartyFile) {
        this.thirdPartyFile = thirdPartyFile;
    }

    protected class LicenseMap extends TreeMap<String, SortedSet<String>> {

        private static final long serialVersionUID = 864199843545688069L;

        public SortedSet<String> put(String key, String value) {
            // handle multiple values as a set to avoid duplicates
            SortedSet<String> valueList = get(key);
            if (valueList == null) {
                valueList = new TreeSet<String>();
            }
            if (getLog().isDebugEnabled()) {
                getLog().debug("key:" + key + ",value: " + value);
            }
            valueList.add(value);
            return put(key, valueList);
        }
    }
}