/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.channels.FileChannel;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.maven.model.Resource;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.io.RawInputStreamFacade;

/**
 * Une classe pour mutualiser toutes les méthodes utiles pour un plugin.
 *
 * @author chemit
 */
public class PluginHelper {

    /**
     * Permet de convertir une liste non typee, en une liste typee.
     * <p/>
     * La liste en entree en juste bien castee.
     * <p/>
     * On effectue une verification sur le typage des elements de la liste.
     * <p/>
     * Note : <b>Aucune liste n'est creee, ni recopiee</b>
     *
     * @param <O> le type des objets de la liste
     * @param list la liste a convertir
     * @param type le type des elements de la liste
     * @return la liste typee
     * @throws IllegalArgumentException si un element de la liste en entree n'est
     *                                  pas en adequation avec le type voulue.
     */
    @SuppressWarnings({"unchecked"})
    static public <O> List<O> toGenericList(List<?> list, Class<O> type) throws IllegalArgumentException {
        if (list.isEmpty()) {
            return (List<O>) list;
        }
        for (Object o : list) {
            if (!(type.isAssignableFrom(o.getClass()))) {
                throw new IllegalArgumentException("can not cast List with object of type " + o.getClass() + " to " + type + " type!");
            }
        }
        return (List<O>) list;
    }
    static final protected double[] timeFactors = {1000000, 1000, 60, 60, 24};
    static final protected String[] timeUnites = {"ns", "ms", "s", "m", "h", "d"};

    static public String convertTime(long value) {
        return convert(value, timeFactors, timeUnites);
    }

    static public String convertTime(long value, long value2) {
        return convertTime(value2 - value);
    }

    static public String convert(long value, double[] factors, String[] unites) {
        long sign = value == 0 ? 1 : value / Math.abs(value);
        int i = 0;
        double tmp = Math.abs(value);
        while (i < factors.length && i < unites.length && tmp > factors[i]) {
            tmp = tmp / factors[i++];
        }

        tmp *= sign;
        String result;
        result = MessageFormat.format("{0,number,0.###}{1}", tmp,
                unites[i]);
        return result;
    }

    /**
     * Prefix the lines of the given content with a given prefix.
     *
     * @param prefix prefix to add on each line of text
     * @param prefixForEmpty prefix to add for empty lines
     * @param content the text to treate
     * @return the text transformed
     * @throws IOException if any reading problem
     */
    static public String prefixLines(String prefix, String prefixForEmpty, String content) throws IOException {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new java.io.StringReader(content));
            StringBuilder sb = new StringBuilder();

            String line = reader.readLine();
            while (line != null) {
                line = line.trim();
                if (line.isEmpty()) {
                    sb.append(prefixForEmpty);
                } else {
                    sb.append(prefix);
                    sb.append(line);
                }
                line = reader.readLine();
                if (line != null) {
                    sb.append('\n');
                }
            }

            String result = sb.toString();
            return result;
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    /**
     * Obtenir les clefs de toutes les valeurs nulles ou vide a partir
     * d'un dictionnaire donne.
     *
     * @param map le dictionner a parcourir
     *
     * @return la liste des clefs dont la valeur est nulle ou vide
     */
    static public SortedSet<String> getEmptyKeys(Map<?, ?> map) {
        SortedSet<String> result = new TreeSet<String>();
        for (Entry<?, ?> e : map.entrySet()) {
            if (e.getValue() == null || e.getValue().toString().isEmpty()) {
                result.add(e.getKey().toString());
            }
        }
        return result;
    }

    /**
     * Permet d'avoir les fichiers de proprietes tries.
     *
     * @author julien
     * @author chemit
     */
    public static class SortedProperties extends Properties {

        private static final long serialVersionUID = -1147150444452577558L;
        /** l'encoding a utiliser pour lire et ecrire le properties. */
        protected String encoding;
        /** un drapeau pour savoir s'il faut enlever l'entete generere */
        protected boolean removeHeader;

        public SortedProperties(String encoding) {
            this(encoding, true);
        }

        public SortedProperties(String encoding, boolean removeHeader) {
            super();
            this.encoding = encoding;
            this.removeHeader = removeHeader;
        }

        public SortedProperties(Properties defaults) {
            super(defaults);
        }

        @Override
        public synchronized Enumeration<Object> keys() {
            List<Object> objects = Collections.list(super.keys());
            Vector<Object> result;
            try {
                // Attention, si les clef ne sont pas des string, ca ne marchera pas
                List<String> list = toGenericList(objects, String.class);
                Collections.sort(list);
                result = new Vector<Object>(list);
            } catch (IllegalArgumentException e) {
                // keys are not string !!!
                // can not sort keys
                result = new Vector<Object>(objects);
            }
            return result.elements();
        }

        /**
         * Charge le properties a partir d'un fichier.
         *
         * @param src le fichier src a charger en utilisant l'encoding declare
         * @return l'instance du properties
         * @throws IOException if any io pb
         */
        public SortedProperties load(File src) throws IOException {
            super.load(new InputStreamReader(new FileInputStream(src), encoding));
            return this;
        }

        /**
         * Sauvegarde le properties dans un fichier, sans commentaire et en utilisant l'encoding declare.
         *
         * @param dst the fichier de destination
         * @throws IOException if any io pb
         */
        public void store(File dst) throws IOException {
            if (removeHeader) {
                super.store(new OutputStreamWriter(new PropertiesDateRemoveFilterStream(new FileOutputStream(dst)), encoding), null);
            } else {
                super.store(new FileOutputStream(dst), null);
            }
        }

        /**
         * Sauvegarde le properties dans un fichier, sans commentaire en laissant java encode en unicode.
         *
         * @param dst le fichier de destination
         * @throws IOException if any io pb
         */
        public void store(OutputStream dst) throws IOException {
            if (removeHeader) {
                super.store(new PropertiesDateRemoveFilterStream(dst), null);
            } else {
                super.store(dst, null);
            }
        }
    }

    /**
     * Un ecrivain qui supprime la premiere ligne rencontree dans le flux.
     *
     *
     * <b>Note: </b> Attention, les performance d'utilisation de cet ecrivain
     * est problèmatique, car sur de gros fichiers (>1000 entrees) les
     * performances se degradent serieusement : pour 1200 entrees on arrive à
     * plus de 5 secondes, alors que sans on a 76 ms! ...
     *
     * FIXME : implanter quelque chose de plus performant dans tous les cas
     */
    public static class PropertiesDateRemoveFilterStream extends FilterOutputStream {

        private boolean firstLineOver;
        char endChar;

        public PropertiesDateRemoveFilterStream(OutputStream out) {
            super(out);
            firstLineOver = false;
            String lineSeparator = System.getProperty("line.separator");
            endChar = lineSeparator.charAt(lineSeparator.length() - 1);
        }

        @Override
        public void write(int b) throws IOException {
            if (!firstLineOver) {
                char c = (char) b;
                if (c == endChar) {
                    firstLineOver = true;
                }
            } else {
                out.write(b);
            }
        }
    }

    public static boolean addResourceDir(String newresourceDir, MavenProject project) {
        List<?> resources = project.getResources();
        boolean added = addResourceDir(newresourceDir, project, resources);
        return added;
    }

    public static boolean addTestResourceDir(String newresourceDir, MavenProject project) {
        List<?> resources = project.getTestResources();
        boolean added = addResourceDir(newresourceDir, project, resources);
        return added;
    }

    public static boolean addResourceDir(String newresourceDir, MavenProject project, List<?> resources) {
        boolean shouldAdd = true;
        for (Object o : resources) {
            Resource r = (Resource) o;
            if (!r.getDirectory().equals(newresourceDir)) {
                continue;
            }
            r.addInclude("**/*.properties");
            r.addInclude("**/*.txt");
            shouldAdd = false;
            break;
        }
        if (shouldAdd) {
            Resource r = new Resource();
            r.setDirectory(newresourceDir);
            r.addInclude("**/*.properties");
            r.addInclude("**/*.txt");
            project.addResource(r);
        }
        return shouldAdd;
    }

    /**
     * Permet de copier le fichier source vers le fichier cible.
     *
     * @param source le fichier source
     * @param target le fichier cible
     * @throws java.io.IOException Erreur de copie
     */
    public static void copy(File source, File target) throws IOException {
        target.getParentFile().mkdirs();
        FileChannel sourceChannel = new FileInputStream(source).getChannel();
        FileChannel targetChannel = new FileOutputStream(target).getChannel();
        sourceChannel.transferTo(0, sourceChannel.size(), targetChannel);
        // or
        //  targetChannel.transferFrom(sourceChannel, 0, sourceChannel.size());
        sourceChannel.close();
        targetChannel.close();
    }

    /**
     * Permet de lire un fichier et de retourner sont contenu sous forme d'une
     * chaine de carateres
     *
     * @param file     le fichier a lire
     * @param encoding encoding to read file
     * @return the content of the file
     * @throws IOException if IO pb
     */
    static public String readAsString(File file, String encoding) throws IOException {
        FileInputStream inf = new FileInputStream(file);
        BufferedReader in = new BufferedReader(new InputStreamReader(inf, encoding));
        try {
            return IOUtil.toString(in);
//            return FileUtil.readAsString(r);
        } finally {
            in.close();
        }
    }

    /**
     * Permet de lire un fichier et de retourner sont contenu sous forme d'une
     * chaine de carateres
     *
     * @param file le reader a lire
     * @return the content of the file
     * @throws IOException if IO pb
     */
    static public String readAsString(java.io.Reader file) throws IOException {
        StringBuffer result = new StringBuffer();
        char[] cbuf = new char[2000];
        BufferedReader in = new BufferedReader(file);
        int nb = in.read(cbuf);
        while (nb != -1) {
            result.append(cbuf, 0, nb);
            nb = in.read(cbuf);
        }
        in.close();
        return result.toString();
    }

    /**
     * Sauvegarde un contenu dans un fichier.
     *
     * @param file     le fichier a ecrire
     * @param content  le contenu du fichier
     * @param encoding l'encoding d'ecriture
     * @throws IOException if IO pb
     */
    static public void writeString(File file, String content, String encoding) throws IOException {
        file.getParentFile().mkdirs();
        BufferedWriter out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), encoding));
        out.write(content);
        out.close();
    }

    public static void write(File destFile, String content, String encoding) throws IOException {

        InputStream in = new ByteArrayInputStream(content.getBytes(encoding));
        RawInputStreamFacade facade = new RawInputStreamFacade(in);

        FileUtils.copyStreamToFile(facade, destFile);
    }

    public static List<File> getIncludedFiles(File dir, String[] includes, String[] excludes) {
        DirectoryScanner ds = new DirectoryScanner();
        List<File> result = new ArrayList<File>();
        ds.setBasedir(dir);
        if (includes != null) {
            ds.setIncludes(includes);
        }
        if (excludes != null) {

            ds.setExcludes(excludes);
        }
        ds.addDefaultExcludes();
        ds.scan();
        for (String file : ds.getIncludedFiles()) {
            File in = new File(dir, file);
            result.add(in);
        }
        return result;
    }

    public static void copyFiles(File src, File dst, String[] includes, String[] excludes, boolean overwrite) throws IOException {
        PluginIOContext c = new PluginIOContext();
        c.setInput(src);
        c.setOutput(dst);
        copyFiles(c, includes, excludes, overwrite);
    }

    public static void copyFiles(PluginIOContext p, String[] includes, String[] excludes, boolean overwrite) throws IOException {
        DirectoryScanner ds = new DirectoryScanner();


        for (File input : p.getInputs()) {
            ds.setBasedir(input);
            if (includes != null) {
                ds.setIncludes(includes);
            }
            if (excludes != null) {

                ds.setExcludes(excludes);
            }
            ds.addDefaultExcludes();
            ds.scan();
            for (String file : ds.getIncludedFiles()) {
                File in = new File(input, file);
                File out = new File(p.getOutput(), file);
                if (overwrite) {
                    FileUtils.copyFile(in, out);
                } else {
                    FileUtils.copyFileIfModified(in, out);
                }
            }
        }
    }

    public static void expandFiles(PluginIOContext p, String[] includes, String[] excludes, String[] zipIncludes, boolean overwrite) throws IOException {

        DirectoryScanner ds = new DirectoryScanner();

        for (File input : p.getInputs()) {
            ds.setBasedir(input);
            if (includes != null) {
                ds.setIncludes(includes);
            }
            if (excludes != null) {

                ds.setExcludes(excludes);
            }
            ds.addDefaultExcludes();
            ds.scan();
            for (String file : ds.getIncludedFiles()) {
                File in = new File(input, file);
                File out = new File(p.getOutput(), file).getParentFile();
                expandFile(in, out, zipIncludes, overwrite);
            }
        }
    }

    public static void expandFile(File src, File dst, String[] includes, boolean overwrite) throws IOException {
//        System.out.println("expandFile src:" + src + " to " + dst);
        ZipFile zipFile = new ZipFile(src);
        Enumeration<? extends ZipEntry> entries = zipFile.entries();
        while (entries.hasMoreElements()) {
            ZipEntry nextElement = entries.nextElement();
            String name = nextElement.getName();
//            System.out.println("name : " + name);
            for (String include : includes) {
                if (DirectoryScanner.match(include, name)) {
                    System.out.println("matching name : " + name + " with pattern " + include);
                    File dstFile = new File(dst, name);
                    if (overwrite || !dstFile.exists() || nextElement.getTime() > dstFile.lastModified()) {
                        System.out.println("will expand : " + name + " to " + dstFile);
                        InputStream inputStream = zipFile.getInputStream(nextElement);
                        FileOutputStream outStream = new FileOutputStream(dstFile);
                        IOUtil.copy(inputStream, outStream, 2048);
                    }

                }
            }
        }
    }
}
