/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

/**
 * Class responsible of instanciate {@link License}.
 * <p/>
 * The factory use a table of {@link #resolvers} to lookup for licenses.
 * <p/>
 * To obtain an instance of a {@link org.nuiton.license.LicenseFactory} use the method {@link #newInstance(String[])}.
 *
 * @author chemit
 */
public class LicenseFactory {

    protected List<LicenseResolver> resolvers;

    public static LicenseFactory newInstance(String... extraResolvers) throws IllegalArgumentException {
        LicenseFactory result = new LicenseFactory();
        // always put jar resolver first
        result.addResolver(new JarLicenseResolver());

        if (extraResolvers != null) {
            for (String extraResolver : extraResolvers) {
                if (extraResolver != null && !extraResolver.trim().isEmpty()) {
                    // add a extra resolver
                    result.addResolver(new LicenseResolver(extraResolver));
                }
            }
        }
        return result;
    }

    public void addResolver(LicenseResolver licenseResolver) {
        resolvers.add(licenseResolver);
    }

    public Map<String, String> getLicenseNames() throws IllegalArgumentException {
        Map<String, String> result = new TreeMap<String, String>();
        for (LicenseResolver resolver : resolvers) {
            Map<String, String> tmp = resolver.getLicenseNames();
            for (Entry<String, String> s : tmp.entrySet()) {
                if (!result.containsKey(s.getKey())) {
                    result.put(s.getKey(), s.getValue());
                }
            }
        }
        return result;
    }


    public License revolv(String licenseName) throws IllegalArgumentException {
        if (licenseName == null) {
            throw new IllegalArgumentException("licenseName can not be null");
        }
        for (LicenseResolver resolver : resolvers) {
            License result = resolver.resolv(licenseName);
            if (result != null) {
                return result;
            }
        }
        throw new IllegalArgumentException("could not find the license " + licenseName + " with resolvers" + resolvers);
    }

    protected LicenseFactory() throws IllegalArgumentException {
        this.resolvers = new ArrayList<LicenseResolver>();
    }

}
