/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton;

import java.io.File;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import org.apache.maven.project.MavenProject;

/**
 * @goal runJava
 * @requiresDependencyResolution test
 * @requiresProject true
 * @execute phase="process-classes"
 * @since 1.0
 * @author chemit
 */
public class RunJavaPlugin extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * The main class to execute.
     *
     * @parameter expression="${mainClass}" default-value="${maven.jar.main.class}"
     * @required
     * @since 1.0
     */
    protected String mainClass;
    /**
     * The class arguments.
     *
     * @parameter expression="${args}"
     * @since 1.0
     */
    protected String args;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${helper.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;
    protected Class<?> clazz;
    protected ClassLoader cl;

    public RunJavaPlugin() {
        super("");
    }

    @Override
    protected boolean ensurePackaging() {
        return project != null && ("pom".equals(project.getPackaging()) || "site".equals(project.getPackaging()));
    }

    @Override
    protected boolean init() throws Exception {

        cl = initClassLoader(project, new File(project.getBuild().getOutputDirectory()), false, false, false, true, true);

        Thread.currentThread().setContextClassLoader(cl);

        clazz = Class.forName(mainClass, true, cl);

        if (args == null) {
            args = "";
        }
        return true;
    }

    @Override
    protected void doAction() throws Exception {
        ThreadGroup threadGroup = Thread.currentThread().getThreadGroup();
        Thread bootstrapThread = new Thread(threadGroup, new Runnable() {

            @Override
            public void run() {
                try {
                    Method main = Class.forName(mainClass, true, cl).getMethod("main", String[].class);
                    if (!main.isAccessible()) {
                        getLog().debug("Setting accessibility to true in order to invoke main().");
                        main.setAccessible(true);
                    }
                    main.invoke(main, new Object[]{args.split(" ")});
                } catch (NoSuchMethodException e) {
                    // just pass it on
                    Thread.currentThread().getThreadGroup().uncaughtException(Thread.currentThread(), new Exception("The specified mainClass doesn't contain a main method with appropriate signature.", e));
                } catch (Exception e) {
                    // just pass it on
                    Thread.currentThread().getThreadGroup().uncaughtException(Thread.currentThread(), e);
                } finally {
                    
                }
            }
        }, mainClass + ".main()");

        bootstrapThread.setContextClassLoader(cl);

        bootstrapThread.start();

        joinNonDaemonThreads(threadGroup);
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    private void joinNonDaemonThreads(ThreadGroup threadGroup) {
        boolean foundNonDaemon;
        do {
            foundNonDaemon = false;
            Collection<Thread> threads = getActiveThreads(threadGroup);
            for (Iterator<Thread> iter = threads.iterator(); iter.hasNext();) {
                Thread thread = iter.next();
                if (thread.isDaemon()) {
                    continue;
                }
                foundNonDaemon = true;   //try again; maybe more threads were created while we were busy
                joinThread(thread, 0);
            }
        } while (foundNonDaemon);
    }

    private void joinThread(Thread thread, long timeoutMsecs) {
        try {
            getLog().debug("joining on thread " + thread);
            thread.join(timeoutMsecs);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();   // good practice if don't throw
            getLog().warn("interrupted while joining against thread " + thread, e);   // not expected!
        }
        if (thread.isAlive()) //generally abnormal
        {
            getLog().warn("thread " + thread + " was interrupted but is still alive after waiting at least " + timeoutMsecs + "msecs");
        }
    }

    private Collection<Thread> getActiveThreads(ThreadGroup threadGroup) {
        Thread[] threads = new Thread[threadGroup.activeCount()];
        int numThreads = threadGroup.enumerate(threads);
        Collection<Thread> result = new ArrayList<Thread>(numThreads);
        for (int i = 0; i < threads.length && threads[i] != null; i++) {
            result.add(threads[i]);
        }
        return result; //note: result should be modifiable
    }

}
