package org.nuiton.jaxx.runtime.api.context;

import org.nuiton.jaxx.runtime.api.JAXXObject;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Created on 4/5/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
public class JAXXContexts {

    public static final String PARENT = "parent";

    public static <O> JAXXContextEntryDef<O> newContextEntryDef(
            Class<O> klass) {
        return newContextEntryDef(null, klass);
    }

    public static <O> JAXXContextEntryDef<O> newContextEntryDef(
            String name, Class<?> klass) {
        return new JAXXContextEntryDef<O>(name, (Class<O>) klass);
    }

    public static <O> JAXXContextEntryDef<List<O>> newListContextEntryDef() {
        return newListContextEntryDef(null);
    }

    public static <O> JAXXContextEntryDef<List<O>> newListContextEntryDef(
            String name) {
        Class<List<O>> castList = castList();
        JAXXContextEntryDef<List<O>> contextEntryDef =
                new JAXXContextEntryDef<List<O>>(name, castList);
        return contextEntryDef;
    }

    public static <K, V> JAXXContextEntryDef<Map<K, V>> newMapContextEntryDef(
            String name) {
        JAXXContextEntryDef<Map<K, V>> contextEntryDef =
                new JAXXContextEntryDef<Map<K, V>>(Map.class, name);
        return contextEntryDef;
    }

    /**
     * Method to initialize the context of a ui.
     *
     * @param ui            the ui
     * @param parentContext the context to set in ui
     */
    public static void initContext(JAXXObject ui, JAXXContext parentContext) {

        if (parentContext instanceof JAXXInitialContext) {
            ((JAXXInitialContext) parentContext).to(ui);
        } else {
            ui.setContextValue(parentContext);
        }
        // if parentContext is a JAXXObject then
        // add an special parent entry to can go up
        if (parentContext instanceof JAXXObject) {
            ui.setContextValue(parentContext, PARENT);
        }
    }

    private static <O> Class<List<O>> castList() {
        return (Class<List<O>>) Collections.emptyList().getClass();
    }

    /**
     * Test if some entries exists in a given context and throw an
     * IllegalArgumentException if not.
     * <p/>
     *
     * @param context the context to test
     * @param defs    the definitions of entries to seek in context
     * @throws IllegalArgumentException if the entry is not found in context.
     */
    public static void checkJAXXContextEntries(JAXXContext context,
                                               JAXXContextEntryDef<?>... defs)
            throws IllegalArgumentException {

        for (JAXXContextEntryDef<?> def : defs) {
            Object value = def.getContextValue(context);

            if (value == null) {
                throw new IllegalArgumentException(
                        "the context entry [" + def + "] ] was not found in " +
                        "context " + context);
            }
        }
    }

    /**
     * Test if a type of entry exists in a given context and throw an
     * IllegalArgumentException if not found.
     * <p/>
     * If entry is found, return his value in context.
     *
     * @param <T>     the type of required data
     * @param context the context to test
     * @param def     the definition of the entry to seek in context
     * @return the value from the context
     * @throws IllegalArgumentException if the entry is not found in context.
     */
    public static <T> T checkJAXXContextEntry(JAXXContext context,
                                              JAXXContextEntryDef<T> def)
            throws IllegalArgumentException {

        T value = def.getContextValue(context);

        if (value == null) {
            throw new IllegalArgumentException(
                    "the context entry [" + def + "] ] was not found in " +
                    "context " + context);
        }

        return value;
    }
}
