package jaxx.demo;

/*
 * #%L
 * JAXX :: Demo
 * *
 * $Id: DemoUIHandler.java 2704 2013-07-21 10:57:39Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.27/jaxx-demo/src/main/java/jaxx/demo/DemoUIHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import jaxx.demo.tree.DemoNode;
import jaxx.demo.tree.DemoTreeHelper;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.DefaultApplicationContext;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.config.ConfigUIHelper;
import jaxx.runtime.swing.log.JAXXLog4jUI;
import jaxx.runtime.swing.renderer.DecoratorProviderListCellRenderer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.decorator.DecoratorProvider;

import javax.swing.AbstractAction;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.util.Arrays;
import java.util.Locale;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * Le handler de l'ui principale.
 *
 * @author tchemit <chemit@codelutin.com>
 * @see DemoUI
 */
public class DemoUIHandler { //implements JAXXHelp {

    /** Logger */
    static private Log log = LogFactory.getLog(DemoUIHandler.class);

    static final JAXXContextEntryDef<DemoUI> MAIN_UI_ENTRY_DEF =
            JAXXUtil.newContextEntryDef("mainui", DemoUI.class);

    public static final String OPEN_CONFIG_ACTION = "openConfig";

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param config      la configuration a utiliser
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public DemoUI initUI(DefaultApplicationContext rootContext, DemoConfig config) {

        if (log.isDebugEnabled()) {
            log.debug("fullscreen ? " + config.isFullScreen());
        }
        DecoratorProvider decoratorProvider =
                rootContext.getContextValue(DecoratorProvider.class);

        if (!config.getLocale().equals(I18n.getDefaultLocale())) {
            if (log.isInfoEnabled()) {
                log.info("re-init I18n with locale " + config.getLocale());
            }
            // change i18n language
            I18n.setDefaultLocale(config.getLocale());
            // reload decorators
            decoratorProvider.reload();
        }

        // create restrict context for ui 
        JAXXInitialContext context = new JAXXInitialContext();

        // share handler
        context.add(this);

        // share tree helper
        context.add(rootContext.getContextValue(DemoTreeHelper.class));

        // share config
        context.add(rootContext.getContextValue(DemoConfig.class));

        // share a unique DecoratorProvider
        context.add(decoratorProvider);

        // share a unique DecoratorProviderListCellRenderer
        context.add(new DecoratorProviderListCellRenderer(decoratorProvider));

        // instanciate ui
        DemoUI ui = new DemoUI(context);

        // add key strokes
        addKeyStrokes(ui, config);

        // keep it in root context
        MAIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);

        // synch to error dialog
        ErrorDialogUI.init(ui);

        // set fullscreen propery on main ui
        ui.getGraphicsConfiguration().getDevice().setFullScreenWindow(config.isFullScreen() ? ui : null);

        return ui;
    }

    protected void addKeyStrokes(final DemoUI ui, DemoConfig config) {

        // Use WHEN_IN_FOCUSED_WINDOW to don't have focus check for binding keys
        JPanel mainPane = ui.getMainPane();
        mainPane.getActionMap().put(OPEN_CONFIG_ACTION, new AbstractAction() {

            @Override
            public void actionPerformed(ActionEvent e) {
                showConfig(ui);
            }
        });
        final InputMap inputMap = mainPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        config.addPropertyChangeListener(DemoConfig.PROPERTY_KEY_OPEN_CONFIG, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                KeyStroke oldValue = (KeyStroke) evt.getOldValue();
                inputMap.remove(oldValue);
                setShowConfigInputMap(inputMap, (KeyStroke) evt.getNewValue());
            }
        });
        setShowConfigInputMap(inputMap, config.getKeyOpenConfig());
    }

    protected void setShowConfigInputMap(InputMap inputMap, KeyStroke keyStroke) {
        inputMap.put(keyStroke, OPEN_CONFIG_ACTION);
    }

    public void initUI(final DemoUI ui) {

        // Creation of selection listener to open ui when tree selection change
        TreeSelectionListener listener = new TreeSelectionListener() {
            @Override
            public void valueChanged(TreeSelectionEvent event) {
                TreePath path = event.getPath();
                DemoNode node = (DemoNode) path.getLastPathComponent();

                if (log.isDebugEnabled()) {
                    log.debug("Select node " + node);
                }

                if (node == null || node.isStringNode()) {

                    // noeud de présentation, rien a faire
                    return;
                }
                showUI(ui, node.getInternalClass());
            }
        };

        JTree tree = ui.getNavigation();

        ui.getTreeHelper().setUI(tree, true, listener);

        // auto-expand node when selected
        SwingUtil.addExpandOnClickListener(tree);

    }

    protected void showUI(DemoUI ui, Class<?> type) {

        JPanel content = ui.getContent();

        String constraints = type.getName();

        log.info("Show for " + constraints);

        // Verify if instance is existing
        DemoTab demoContent = getContentIfExist(ui, constraints);

        if (demoContent == null) {

            log.info("Will instanciate a new " + constraints);
            try {
                // Get constructor
                Constructor<?> constructor =
                        type.getConstructor(JAXXContext.class);

                JAXXInitialContext tx =
                        new JAXXInitialContext().add(ui.getDelegateContext());

                DemoPanel panel = (DemoPanel) constructor.newInstance(tx);

                log.info("Demo panel to use : " + panel);

                demoContent = new DemoTab(tx.add("content", panel));

            } catch (Exception eee) {
                log.error(eee, eee);
                ErrorDialogUI.showError(eee);
            }

            // Add to content panel
            content.add(demoContent, constraints);
        }

        // show ui
        ui.getContentLayout().show(content, constraints);
    }

// Get content if exist in content, else return null

    protected <E extends Component> E getContentIfExist(DemoUI ui, String constraints) {
        if (log.isDebugEnabled()) {
            log.debug("Get content if exist " + constraints);
        }
        if (!ui.getContentLayout().contains(constraints)) {
            return null;
        }
        return (E) ui.getContentLayout().getComponent(ui.getContent(), constraints);
    }

    public void displayUI(final DemoUI ui, final String[] nodePath) {

        // expend all nodes of the demo tree
        SwingUtil.expandTree(ui.getNavigation());

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {

                String[] path;
                if (nodePath == null) {
                    // take first node
                    path = new String[]{"jaxxdemo.tree"};
                } else {
                    // take selected node
                    path = nodePath;

                }
                log.info("node to re select " + Arrays.toString(path));
                // select node
                ui.getTreeHelper().selectNode(path);

                // use best dimensions
                ui.getSplitPane().resetToPreferredSizes();
            }
        });

        // show ui after all (in another invocation, tu avoid layout adjustement
        // to be seen).

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {

                // show ui
                ui.setVisible(true);
            }
        });
    }

    /**
     * Permet de recharger l'ui principale et de changer de le mode d'affichage.
     *
     * @param rootContext le contexte applicatif
     * @param config      la configuration a utiliser
     */
    public void reloadUI(DefaultApplicationContext rootContext, DemoConfig config) {

        // scan main ui
        DemoUI ui = getUI(rootContext);

        String[] node = null;
        if (ui != null) {

            ui.getConfig().removeJaxxPropertyChangeListener();

            node = ui.getTreeHelper().getSelectedIds();

            if (node != null) {
                if (log.isDebugEnabled()) {
                    log.debug("selected node " + Arrays.toString(node));
                }
            }

            ErrorDialogUI.init(null);

            ui.dispose();

            ui.setVisible(false);

            MAIN_UI_ENTRY_DEF.removeContextValue(rootContext);
        }

        ui = initUI(rootContext, config);

        displayUI(ui, node);
    }

    /**
     * Méthode pour changer de mode d'affichage.
     * <p/>
     * Si <code>fullscreen</code> est à <code>true</code> alors on passe en
     * mode console (c'est à dire en mode plein écran exclusif), sinon on
     * passe en mode fenetré normal.
     *
     * @param context    l'ui principale de l'application
     * @param fullscreen le nouvel état requis.
     */
    public void changeScreen(JAXXContext context, boolean fullscreen) {

        DemoUI ui = getUI(context);

        // sauvegarde de l'état dans la configuration
        DemoConfig config = ui.getConfig();
        config.setFullscreen(fullscreen);

        // rechargement de l'ui
        reloadUI(RunDemo.get(), config);
    }

    public void changeLanguage(JAXXContext context, Locale newLocale) {

        DemoUI ui = getUI(context);

        DemoConfig config = ui.getConfig();

        // sauvegarde de la nouvelle locale
        config.setLocale(newLocale);

        // rechargement de l'ui
        reloadUI(RunDemo.get(), config);
    }

    /**
     * Ferme l'application.
     *
     * @param context l'ui principale de l'application
     */
    public void close(JAXXContext context) {
        log.info("JAXX Demo quitting...");
        try {

            DemoUI ui = getUI(context);
            ui.dispose();
        } finally {
            System.exit(0);
        }
    }

    final Runnable reloadUICallback = new Runnable() {

        @Override
        public void run() {
            if (log.isInfoEnabled()) {
                log.info("will reload ui");
            }
            DefaultApplicationContext context = RunDemo.get();
            DemoUI ui = getUI(context);
            DemoConfig config = ui.getConfig();
            reloadUI(context, config);
        }
    };

    final Runnable reloadApplicationCallback = new Runnable() {

        @Override
        public void run() {
            if (log.isInfoEnabled()) {
                log.info("will reload appplication");
            }
            close(RunDemo.get());
        }
    };

    final Runnable reloadLogAppenderCallback = new Runnable() {

        @Override
        public void run() {
            if (log.isInfoEnabled()) {
                log.info("will reload log appender");
            }
            DefaultApplicationContext context = RunDemo.get();
            DemoUI ui = getUI(context);
            DemoConfig config = ui.getConfig();

            // init jaxx logger
            JAXXLog4jUI.init(config.getLogLevel(), config.getLogPatternLayout());
        }
    };

    public void showConfig(JAXXContext context) {
        DemoUI ui = getUI(context);
        DemoConfig config = ui.getConfig();

        ConfigUIHelper helper = new ConfigUIHelper(config);

        helper.registerCallBack("ui",
                                n_("demo.action.reload.ui"),
                                SwingUtil.createActionIcon("reload-ui"),
                                reloadUICallback);

        helper.registerCallBack("application",
                                n_("demo.action.reload.application"),
                                SwingUtil.createActionIcon("reload-application"),
                                reloadApplicationCallback);

        helper.registerCallBack("log",
                                n_("demo.action.reload.logAppender"),
                                SwingUtil.createActionIcon("reload-log"),
                                reloadLogAppenderCallback);

        // categorie repertoires

        helper.addCategory(n_("jaxxdemo.config.category.directories"),
                           n_("jaxxdemo.config.category.directories.description"));

        helper.addOption(DemoConfig.Option.CONFIG_FILE);

        // others
        helper.addCategory(n_("jaxxdemo.config.category.other"),
                           n_("jaxxdemo.config.category.other.description"));

        helper.addOption(DemoConfig.Option.FULL_SCREEN,
                         DemoConfig.PROPERTY_FULLSCREEN);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.FONT_SIZE,
                         DemoConfig.PROPERTY_FONT_SIZE);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.LOCALE,
                         DemoConfig.PROPERTY_LOCALE);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.DEMO_COLOR,
                         DemoConfig.PROPERTY_DEMO_COLOR);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.DEMO_CLASS,
                         DemoConfig.PROPERTY_DEMO_CLASS);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.KEY_OPEN_CONFIG,
                         DemoConfig.PROPERTY_KEY_OPEN_CONFIG);
        helper.setOptionCallBack("ui");

        helper.addOption(DemoConfig.Option.LOG_LEVEL,
                         DemoConfig.PROPERTY_LOG_LEVEL);
        helper.setOptionCallBack("log");

        helper.addOption(DemoConfig.Option.LOG_PATTERN_LAYOUT,
                         DemoConfig.PROPERTY_LOG_PATTERN_LAYOUT);
        helper.setOptionCallBack("log");

        helper.buildUI(context, "jaxxdemo.config.category.other");

        helper.displayUI(ui, false);
    }

    public void showHelp(JAXXContext context, String helpId) {
        log.info(context + " :: " + helpId);
//        DemoUI mainUI = getUI(context);
//        ObserveHelpBroker helpBroker = context.getContextValue(ObserveHelpBroker.class);
//
//        if (helpId == null) {
//            helpId = helpBroker.getDefaultID();
//        }
//        log.debug("show help " + helpId);
//        mainUI.getHelp().setCurrentID(helpId);
    }

    public void showLogs(DemoUI ui) {

        JAXXLog4jUI log4jUI = new JAXXLog4jUI();
        String title = _("jaxxdemo.title.showLog");
        log4jUI.setTitle(title);
        log4jUI.showInDialog(ui, false);
    }

    public void gotoSite(JAXXContext context) {

        DemoUI ui = getUI(context);
        DemoConfig config = ui.getConfig();

        URL siteURL = config.getApplicationSiteUrl();

        log.info(_("jaxxdemo.message.goto.site", siteURL));

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        if (Desktop.isDesktopSupported() && Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                ErrorDialogUI.showError(ex);
            }
        }
    }

    public void showAbout(DemoUI ui) {

        DemoConfig config = ui.getConfig();

        String iconPath = config.getIconPath();
        String licensePath = config.getLicensePath();
        String thirdPartyPath = config.getThirdParty();

        AboutPanel about = new AboutPanel();
        about.setTitle(_("jaxxdemo.title.about"));
        about.setAboutText(_("jaxxdemo.about.message"));
        about.setBottomText(ui.getConfig().getCopyrightText());
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();
        about.init();
        about.showInDialog(ui, true);
    }

    DemoUI getUI(JAXXContext context) {
        if (context instanceof DemoUI) {
            return (DemoUI) context;
        }
        DemoUI ui = MAIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }
}
