/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: HelpRootCompiledObjectDecorator.java 2055 2010-08-30 10:01:11Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.1/jaxx-compiler/src/main/java/jaxx/compiler/decorators/HelpRootCompiledObjectDecorator.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.decorators;

import jaxx.compiler.CompiledObject;
import jaxx.compiler.CompilerConfiguration;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.java.JavaArgument;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.runtime.swing.help.JAXXHelpUI;

import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * A decorator to place on a root compiled object to process javaHelp on the file.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role-hint="help" role="jaxx.compiler.CompiledObjectDecorator"
 * @since 1.2
 */
public class HelpRootCompiledObjectDecorator extends DefaultCompiledObjectDecorator {

    /** the list of discovered helpId */
    protected static Set<String> helpIds = new HashSet<String>();

    @Override
    public String getName() {
        return "help";
    }

    protected String getBrokerFQN(JAXXCompiler compiler) {
        String helpBrokerFQN = compiler.getConfiguration().getHelpBrokerFQN();
        return helpBrokerFQN;
    }

    protected String getHelpId(CompiledObject o) {
        String helpID = null;
        if (o.hasClientProperties()) {
            helpID = o.getClientProperty("help");
        }
        return helpID;
    }

    @Override
    public void finalizeCompiler(JAXXCompiler compiler,
                                 CompiledObject root,
                                 CompiledObject object,
                                 JavaFile javaFile,
                                 String packageName,
                                 String className,
                                 String fullClassName) {
        super.finalizeCompiler(compiler,
                               root,
                               object,
                               javaFile,
                               packageName,
                               className,
                               fullClassName
        )
                ;
        CompilerConfiguration options = compiler.getConfiguration();

        if (options.isGenerateHelp()) {

            // add JAXXHelpUI interface
            Class<?> validatorInterface = JAXXHelpUI.class;
            String helpBrokerFQN = getBrokerFQN(compiler);
            javaFile.addInterface(
                    JAXXCompiler.getCanonicalName(validatorInterface) + "<" +
                    helpBrokerFQN + ">");

            javaFile.addMethod(JavaFileGenerator.newMethod(
                    Modifier.PUBLIC,
                    "void",
                    "registerHelpId",
                    "broker.installUI(component, helpId);",
                    true,
                    new JavaArgument(helpBrokerFQN, "broker"),
                    new JavaArgument("Component", "component"),
                    new JavaArgument("String", "helpId"))
            );

            javaFile.addMethod(JavaFileGenerator.newMethod(
                    Modifier.PUBLIC,
                    "void",
                    "showHelp",
                    "getBroker().showHelp(this, helpId);",
                    true,
                    new JavaArgument("String", "helpId"))
            );

            StringBuilder buffer = new StringBuilder();

            String eol = JAXXCompiler.getLineSeparator();

//            if (options.isGenerateHelp()) {

            // add code to init javax help system
            Iterator<CompiledObject> itr = compiler.getObjectCreationOrder();

            for (; itr.hasNext();) {
                CompiledObject o = itr.next();
                String helpID = getHelpId(o);
                if (helpID != null) {
                    buffer.append(eol);
                    // detects a helpId to register
                    buffer.append("registerHelpId(_broker, ");
                    buffer.append(o.getJavaCode());
                    buffer.append(", ");
                    buffer.append(helpID);
                    buffer.append(");");
                    //keep the helpID for helpSet generation
                    helpIds.add(helpID);
                }
            }
//            }
            if (buffer.length() > 0) {

                StringBuilder extraCode = new StringBuilder(helpBrokerFQN);
                extraCode.append(" _broker = getBroker();");

                buffer.append(eol).append("_broker.prepareUI(this);");
                buffer.append(eol);

                // add the calls
                compiler.appendLateInitializer("// help broker setup" + eol);
                compiler.appendLateInitializer(extraCode.toString());
                compiler.appendLateInitializer(buffer.toString());
            }
        }
    }

    public static Set<String> getHelpIds() {
        return new HashSet<String>(helpIds);
    }
}
