/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler;

import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.spi.Initializer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

/**
 * @author chemit
 * @since 2.0.0 was previously JAXXCompilerLaunchor
 */
public class JAXXEngine {

    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(JAXXEngine.class);

    public void addJaxxFileClassName(String className) {
        jaxxFileClassNames.add(className);
    }

    public void addJaxxFile(File jaxxFile) {
        jaxxFiles.add(jaxxFile);
    }

    public boolean containsJaxxFileClassName(String className) {
        return jaxxFileClassNames.contains(className);
    }

    public LifeCycle getCurrentPass() {
        return currentPass;
    }
    /** shared instance of unique launchor at a givne time. */
    protected static JAXXEngine singleton;

    /**
     * Create a new empty launchor and set it as current launchor accessible via method {@link #get()}
     *
     * @return the new instanciated launchor
     */
    public static synchronized JAXXEngine newLaunchor() {
        return newLaunchor((File[]) null, null, null);
    }

    /**
     * Create a new launchor and set it as current launchor accessible via method {@link #get()}.
     * <p/>
     * The launchor will be prepared to run a set of files, expressed as paths relative to a base directory.
     * The class names of the compiled files are derived from the relative path strings
     * (e.g. "example/Foo.jaxx" compiles into a class named "example.Foo").
     *
     * @param base          the directory against which to resolve relative paths
     * @param relativePaths a list of relative paths to .jaxx files being compiled
     * @param configuration       the compiler configuration to use
     * @return the new instanciated launchor
     */
    public static synchronized JAXXEngine newLaunchor(File base, String[] relativePaths, CompilerConfiguration configuration) {
        File[] files = new File[relativePaths.length];
        String[] classNames = new String[relativePaths.length];
        for (int i = 0; i < files.length; i++) {
            files[i] = new File(base, relativePaths[i]);
            classNames[i] = relativePaths[i].substring(0, relativePaths[i].lastIndexOf("."));
            classNames[i] = classNames[i].replace(File.separatorChar, '.');
            classNames[i] = classNames[i].replace('/', '.');
            classNames[i] = classNames[i].replace('\\', '.');
            classNames[i] = classNames[i].replace(':', '.');
        }
        return newLaunchor(files, classNames, configuration);
    }

    /**
     * Create a new launchor and set it as current launchor accessible via method {@link #get()}.
     * <p/>
     * The launchor will be prepared to run a set of files, with the class names specified explicitly.
     * The class compiled from files[i] will be named classNames[i].
     *
     * @param files      the .jaxx files to run
     * @param classNames the names of the classes being compiled
     * @param configuration    the compiler configuration to use
     * @return the new instanciated launchor
     */
    public static synchronized JAXXEngine newLaunchor(File[] files, String[] classNames, CompilerConfiguration configuration) {
        if (singleton != null) {
            singleton.reset();
        }
        singleton = new JAXXEngine(files, classNames, configuration);
        return singleton;
    }

    /**
     * @return the current launchor
     * @throws NullPointerException if no launchor was registred via a <code>newLaunchor-like</code> method.
     */
    public static JAXXEngine get() throws NullPointerException {
        if (singleton == null) {
            throw new NullPointerException("no launchor was registred via newLaunchor method");
        }
        return singleton;
    }

    /** @return <code> if there is a launchor registred, <code>false</code> otherwise. */
    public static boolean isRegistred() {
        return singleton != null;
    }

    /**
     * Load the {@link Initializer} services found via the{@link ServiceLoader} mecanism.
     *
     * @param verbose <ocde>true</code> to print initializers
     */
    public static void loadLibraries(boolean verbose) {
        //BeanInfoUtil.reset();
        ClassLoader classloader = Thread.currentThread().getContextClassLoader();
        if (verbose) {
            log.info("with cl " + classloader);
        }
        ServiceLoader<Initializer> loader = ServiceLoader.load(Initializer.class, classloader);
        for (Initializer initializer : loader) {
            if (verbose) {
                log.info("load initializer " + initializer);
            }
            initializer.initialize();
        }
    }
    /**
     * configuration of the launchor and underlines compilers
     */
    protected CompilerConfiguration configuration;
    /**
     * original list of files to run
     */
    protected final File[] files;
    /**
     * original list of classes to run
     */
    protected final String[] classNames;
    /**
     * Files to be treated while compilation.
     */
    private List<File> jaxxFiles = new ArrayList<File>();
    /**
     * Class names corresponding to the files in the jaxxFiles list.
     */
    private List<String> jaxxFileClassNames = new ArrayList<String>();
    /**
     * Maps the names of classes being compiled to the compiler instance handling the compilation.
     */
    protected Map<String, JAXXCompiler> compilers = new HashMap<String, JAXXCompiler>();
    /**
     * Maps the names of classes being compiled to their symbol tables (created after the first compiler pass).
     */
    protected Map<File, SymbolTable> symbolTables = new HashMap<File, SymbolTable>();
    /**
     * current pass of the engine
     */
    private LifeCycle currentPass;
    /**
     * Warnings detected while running.
     */
    protected List<String> warnings = new ArrayList<String>();
    /**
     * Errors detected while running.
     */
    protected List<String> errors = new ArrayList<String>();
    /**
     * profile attached to the engine (can be null)
     */
    protected JAXXProfile profiler;
    /**
     * decorators available in engine
     */
    protected Map<String, CompiledObjectDecorator> decorators;
    /**
     * finalizers available in engine
     */
    protected List<JAXXCompilerFinalizer> finalizers;

    /**
     * Different passes of the engine.
     *
     * Each pass defines the {@link #run(JAXXEngine)} method which implements
     * the logic of the pass.
     */
    protected enum LifeCycle {

        /**
         * state before compilation
         */
        init {

            @Override
            public boolean run(JAXXEngine engine) {
                boolean success = true;
                engine.warnings.clear();
                engine.errors.clear();
                // init decorators
                if (engine.decorators == null) {
                    engine.decorators = new TreeMap<String, CompiledObjectDecorator>();
                    // load decorators
                    for (CompiledObjectDecorator decorator : ServiceLoader.load(CompiledObjectDecorator.class)) {
                        engine.decorators.put(decorator.getName(), decorator);
                    }
                }
                // init finalizers
                engine.finalizers = new ArrayList<JAXXCompilerFinalizer>();
                for (JAXXCompilerFinalizer finalizer : ServiceLoader.load(JAXXCompilerFinalizer.class)) {
                    engine.finalizers.add(finalizer);
                }

                engine.jaxxFiles.addAll(Arrays.asList(engine.files));
                engine.jaxxFileClassNames.addAll(Arrays.asList(engine.classNames));
                return success;
            }
        },
        /**
         * first pass of compilation
         */
        compile_first_pass {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                boolean compiled;
                do {
                    compiled = false;
                    assert engine.jaxxFiles.size() == engine.jaxxFileClassNames.size();
                    java.util.Iterator<File> filesIterator = new ArrayList<File>(engine.jaxxFiles).iterator(); // clone it so it can safely be modified while we're iterating
                    java.util.Iterator<String> classNamesIterator = new ArrayList<String>(engine.jaxxFileClassNames).iterator();
                    while (filesIterator.hasNext()) {
                        File file = filesIterator.next();
                        String className = classNamesIterator.next();
                        if (log.isDebugEnabled()) {
                            log.debug("compile first pass for " + className);
                        }
                        if (engine.symbolTables.get(file) == null) {
                            compiled = true;
                            if (engine.compilers.containsKey(className)) {
                                throw new CompilerException("Internal error: " + className + " is already being compiled, attempting to compile it again");
                            }

                            File destDir = engine.configuration.getTargetDirectory();
                            if (destDir != null) {
                                int dotPos = className.lastIndexOf(".");
                                if (dotPos != -1) {
                                    destDir = new File(destDir, className.substring(0, dotPos).replace('.', File.separatorChar));
                                }
                                if (!destDir.exists() && !destDir.mkdirs()) {
                                    log.warn("could not create directory " + destDir);
                                    continue;
                                }
                            }
                            JAXXCompiler compiler = engine.newCompiler(file.getParentFile(), file, className);
                            addStartProfileTime(engine, compiler);
                            engine.compilers.put(className, compiler);
                            compiler.compileFirstPass();
                            addEndProfileTime(engine, compiler);
                            assert !engine.symbolTables.values().contains(compiler.getSymbolTable()) : "symbolTable is already registered";
                            engine.symbolTables.put(file, compiler.getSymbolTable());
                            if (compiler.isFailed()) {
                                success = false;
                            }
                        }
                    }

                } while (compiled);
                return success;
            }
        },
        /**
         * second pass of compilation
         */
        compile_second_pass {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                List<File> jaxxFilesClone = new ArrayList<File>(engine.jaxxFiles);
                for (String className : engine.jaxxFileClassNames) {
                    JAXXCompiler compiler = engine.getCompiler(className, "Internal error: could not find compiler for " + className + " during second pass");
                    addStartProfileTime(engine, compiler);
                    if (log.isDebugEnabled()) {
                        log.debug("runInitializers for " + className);
                    }
                    if (!compiler.isFailed()) {
                        compiler.runInitializers();
                    }
                    if (log.isDebugEnabled()) {
                        log.debug("compile second pass for " + className);
                    }
                    compiler.compileSecondPass();
                    addEndProfileTime(engine, compiler);
                    if (log.isDebugEnabled()) {
                        log.debug("done with result [" + !compiler.isFailed() + "] for " + className);
                    }
                    if (compiler.isFailed()) {
                        success = false;
                    }
                }
                if (!jaxxFilesClone.equals(engine.jaxxFiles)) {
                    throw new AssertionError("Internal error: compilation set altered during pass 2 (was " + jaxxFilesClone + ", modified to " + engine.jaxxFiles + ")");
                }
                return success;
            }
        },
        /**
         * applying stylesheet
         */
        stylesheet_pass {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                for (String className : engine.jaxxFileClassNames) {
                    JAXXCompiler compiler = engine.getCompiler(className, "Internal error: could not find compiler for " + className + " during stylesheet application");
                    addStartProfileTime(engine, compiler);
                    compiler.applyStylesheets();
                    addEndProfileTime(engine, compiler);
                    if (compiler.isFailed()) {
                        success = false;
                    }
                }
                return success;
            }
        },
        /**
         * finalize compiler
         */
        finalize_compiler {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                for (String className : engine.jaxxFileClassNames) {
                    JAXXCompiler compiler = engine.getCompiler(className, "Internal error: could not find compiler for " + className + " during code generation");
                    addStartProfileTime(engine, compiler);
                    compiler.finalizeCompiler(engine.finalizers);
                    addEndProfileTime(engine, compiler);
                    if (compiler.isFailed()) {
                        success = false;
                    }
                }
                return success;
            }
        },
        /**
         * generate java file
         */
        generate_pass {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                JavaFileGenerator generator = new JavaFileGenerator(JAXXCompiler.getLineSeparator(), engine.configuration.isVerbose());

                for (String className : engine.jaxxFileClassNames) {
                    JAXXCompiler compiler = engine.getCompiler(className, "Internal error: could not find compiler for " + className + " during code generation");
                    addStartProfileTime(engine, compiler);
                    compiler.generate(generator);
                    addEndProfileTime(engine, compiler);
                    if (compiler.isFailed()) {
                        success = false;
                    }
                }
                return success;
            }
        },
        /**
         * display profile results
         */
        profile_pass {

            @Override
            public boolean run(JAXXEngine engine) throws Exception {
                boolean success = true;
                if (engine.configuration.isProfile()) {
                    StringBuilder buffer = engine.profiler.computeProfileReport();
                    log.info(buffer.toString());
                }
                return success;
            }
        };

        /**
         * Run the pass.
         * 
         * @param engine the engine to use
         * @return {@code true} if pass was ok, {@code false} otherwise
         * @throws Exception if any pb
         */
        public abstract boolean run(JAXXEngine engine) throws Exception;

        void addStartProfileTime(JAXXEngine engine, JAXXCompiler compiler) {
            engine.addProfileTime(compiler, name() + "_start");
        }

        void addEndProfileTime(JAXXEngine engine, JAXXCompiler compiler) {
            engine.addProfileTime(compiler, name() + "_end");
        }
    }

    protected JAXXEngine(File[] files, String[] classNames, CompilerConfiguration options) {
        this.configuration = options == null ? new DefaultCompilerConfiguration() : options;
        this.files = files;
        this.classNames = classNames;
        if (log.isDebugEnabled()) {
            log.debug("files : " + Arrays.toString(files));
        }
        if (this.configuration.isProfile()) {
            profiler = new JAXXProfile();
        }
    }

    /**
     * Resets all state in preparation for a new compilation session.
     */
    protected void reset() {
        jaxxFiles.clear();
        jaxxFileClassNames.clear();
        symbolTables.clear();
        for (JAXXCompiler jaxxCompiler : compilers.values()) {
            jaxxCompiler.clear();
        }
        compilers.clear();
        if (profiler != null) {
            profiler.clear();
            profiler = null;
        }
        if (decorators != null) {
            decorators.clear();
            decorators = null;
        }
        if (finalizers != null) {
            finalizers.clear();
            finalizers = null;
        }
    }

    public String getVersion() {
        return "2.0.0";
    }

    /**
     * Creates a dummy Compiler for use in unit testing or dettached use of an
     * engine.
     *
     * @param classLoader class loader to use
     * @return the compiler
     */
    public static JAXXCompiler createDummyCompiler(final ClassLoader classLoader) {
        JAXXCompiler compiler = new JAXXCompiler(null, null, null, null, null, null);
        compiler.classLoader = classLoader;
        return compiler;
    }

    /**
     * Obtain the jaxx compiler of the given class name.
     * 
     * @param className the name of the class to use
     * @return the compiler instance which is processing the specified JAXX class.  Each class is compiled by a
     *         different compiler instance.
     */
    public JAXXCompiler getJAXXCompiler(String className) {
        return compilers == null ? null : compilers.get(className);
    }

    /**
     * Obtain the symbo table for the given class name.
     *
     * @param className the name of the class to use
     * @return the symbol table for the specified JAXX class.  Must be called during the second compiler pass.
     *         Returns <code>null</code> if no such symbol table could be found.
     */
    public SymbolTable getSymbolTable(String className) {
        JAXXCompiler compiler = getJAXXCompiler(className);
        if (compiler == null) {
            return null;
        }
        return compiler.getSymbolTable();
    }

    /**
     * Obtain the decorator of the given name.
     *
     * @param name the name of the decorator
     * @return the decorator found.
     * @throws IllegalArgumentException if decorator not found for the given name.
     */
    public CompiledObjectDecorator getDecorator(String name) throws IllegalArgumentException {
        CompiledObjectDecorator decorator = decorators.get(name);
        if (decorator == null) {
            throw new IllegalArgumentException("could not find decorator with key " + name + " (known decorators : " + decorators.keySet() + ")");
        }
        return decorator;
    }

    /**
     * Obtain the decorator of the given type.
     *
     * @param type the type of the decorator (syas his fqn)
     * @return the decorator found
     */
    public CompiledObjectDecorator getDecorator(Class<?> type) {
        for (CompiledObjectDecorator decorator : decorators.values()) {
            if (type == decorator.getClass()) {
                return decorator;
            }
        }
        return null;
    }

    /**
     * Add a warning to the engine.
     *
     * @param warning the warning to add
     */
    public void addWarning(String warning) {
        warnings.add(warning);
    }

    /**
     * Add an error to the engine.
     *
     * @param error the error to add
     */
    public void addError(String error) {
        errors.add(error);
    }

    /**
     * @return the errors of the engine
     */
    public List<String> getErrors() {
        return errors;
    }

    /**
     * @return the warnings of the engine
     */
    public List<String> getWarnings() {
        return warnings;
    }

    /**
     * Compiled a set of files.
     *
     * @return {@code -1} if errors appears, the number of generated files
     * otherwise.
     */
    public synchronized int run() {
        try {
            boolean success = true;

            for (LifeCycle state : LifeCycle.values()) {
                if (!success) {
                    // stop as soon as a engine phase failed
                    break;
                }
                currentPass = state;
                if (configuration.isVerbose()) {
                    log.info("'" + state + "' on " + jaxxFiles.size() + " file(s)");
                }
                success = state.run(this);
            }
            return success ? compilers.size() : -1;

            //FIXME : deal better  the exception treatment...
        } catch (CompilerException e) {
            log.error(e.getMessage(),e);
//            System.err.println(e.getMessage());
//            e.printStackTrace();
            return -1;
        } catch (Throwable e) {
            log.error(e.getMessage(),e);
//            e.printStackTrace();
            return -1;
        } finally {
            //TC - 20081018 only reset when no error was detected
            if (configuration.isResetAfterCompile() && errors.isEmpty()) {
                reset();
            }
        }
    }

    /**
     * Obtain the compiler for the given class name.
     * 
     * @param className the class name of the searched compiler
     * @param message the message to throw if compiler not found
     * @return the compiler found
     * @throws CompilerException if compiler not found
     */
    protected JAXXCompiler getCompiler(String className, String message) throws CompilerException {
        JAXXCompiler compiler = compilers.get(className);
        if (compiler == null) {
            throw new CompilerException(message);
        }
        return compiler;
    }

    /**
     * Create a new compiler.
     *
     * @param parentFile the directory where to generate the java file
     * @param file the path of file to generate
     * @param className name of the class to generate in the file
     * @return the new compiler
     * @throws InvocationTargetException introspection error
     * @throws IllegalAccessException introspection error
     * @throws InstantiationException introspection error
     * @throws NoSuchMethodException introspection error
     */
    protected JAXXCompiler newCompiler(File parentFile, File file, String className) throws InvocationTargetException, IllegalAccessException, InstantiationException, NoSuchMethodException {
        Constructor<? extends JAXXCompiler> cons = configuration.getCompilerClass().getConstructor(JAXXEngine.class, File.class, File.class, String.class, CompilerConfiguration.class, List.class);
        return cons.newInstance(this, parentFile, file, className, configuration,
                    Arrays.asList("java.awt.*",
                    "java.awt.event.*",
//                    "java.beans.*",
                    "java.io.*",
                    "java.lang.*",
                    "java.util.*",
                    "javax.swing.*",
                    "javax.swing.border.*",
                    "javax.swing.event.*",
                    "jaxx.runtime.swing.JAXXButtonGroup",
                    "jaxx.runtime.swing.HBox",
                    "jaxx.runtime.swing.VBox",
                    "jaxx.runtime.swing.Table",
                    "jaxx.runtime.Util",
                    "jaxx.runtime.SwingUtil",
                    "static org.nuiton.i18n.I18n._",
                    "static jaxx.runtime.SwingUtil.createImageIcon")
                );
    }

    /**
     * Add a profile time for the given compiler and key.
     *
     * Note: if {@link #profiler} is {@code null}, do nothing
     *
     * @param compiler the compiler to profile
     * @param key the key of profiling
     */
    public void addProfileTime(JAXXCompiler compiler, String key) {
        if (profiler != null) {
            profiler.addTime(compiler, key);
        }
    }
}
