/*
 * #%L
 * I18n :: Api
 * 
 * $Id: I18n.java 1789 2010-11-08 10:28:45Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.0.1/nuiton-i18n/src/main/java/org/nuiton/i18n/I18n.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.init.ClassPathI18nInitializer;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.I18nInitializer;

import java.util.Arrays;
import java.util.Locale;

/**
 * New generation I18n class.
 * <p/>
 * <b>Note:</b> This class replace the previous one in project {@code
 * nuiton-utils}.
 * <p/>
 * This class is a facility for internationalization. To use it in your soft,
 * you can either : <ul> <li> import the org.nuiton.i18n.I18n class, <li> init
 * the translation support with the init(String language) or init(String
 * language, String country), init(Localelocale) static methods in your main, (
 * eg: I18n.init("fr","FR") ) <li> call the translate static method for each
 * sentence, ( eg: I18n._("hello you !") ) <li> create a resource file for each
 * language following the naming convention given in the
 * java.util.ResourceBundle javadoc and translate all the sentence. </ul>
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.1
 */
public class I18n {

    /** Logger */
    private static final Log log = LogFactory.getLog(I18n.class);

    public static final String ISO_8859_1_ENCONDING = "ISO-8859-1";

    public static final String UTF_8_ENCONDING = "UTF-8";

    public static final String DEFAULT_ENCODING = ISO_8859_1_ENCONDING;

    public static final Locale DEFAULT_LOCALE = Locale.UK;

    /** I18n initializer */
    protected static I18nInitializer initializer;

    /** I18n store of languages */
    protected static I18nStore store;

    /** Filtre a appliquer avant de retourner les chaines */
    protected static I18nFilter filter;

    /**
     * Sets the initializer to use to init the I18n system.
     * <p/>
     * You must inovke this method before any invocation of methods {@code
     * init(XXX)}.
     *
     * @param initializer the initializer to set.
     * @see DefaultI18nInitializer
     */
    public static void setInitializer(I18nInitializer initializer) {
        I18n.initializer = initializer;
    }

    /**
     * Change le filtre des chaines traduites
     *
     * @param filter l'objet filtre a utiliser
     */
    public static void setFilter(I18nFilter filter) {
        I18n.filter = filter;
    }

    /** Initialise la librairie avec encoding par defaut et locale par defaut */
    public static void init() {
        init(null);
    }

    /**
     * Initialise la librairie
     *
     * @param language une chaine representant la langue à utiliser fr, en, ...
     * @param country  une chaine representant le pays à utiliser FR, GB, ...
     */
    public static void init(String language, String country) {
        init(I18nUtil.newLocale(language, country));
    }

    /**
     * Initialize the library for given <code>locale</code> with {@link
     * #DEFAULT_ENCODING}.
     *
     * @param locale language to use
     */
    public static void init(Locale locale) {
        if (initializer == null) {
            initializer = new ClassPathI18nInitializer();
        }

        if (locale == null) {
            // use default locale
            locale = I18nUtil.newLocale(null, null);
        }

        // let the store use this locale
        getStore().setLanguage(locale);
    }

    /**
     * Retourne la chaine traduite si possible.
     *
     * @param message la chaine a traduire
     * @return la traduction si possible ou la chaine passee en parametre
     *         sinon.
     */
    public static String _(String message) {

        // if the key to translate is null, just return null
        if (message == null) {
            return null;
        }

        I18nLanguage language = getCurrentLanguage();
        String result = message;
        if (language != null) {
            result = language.translate(message);
        }
        return applyFilter(result);
    }

    /**
     * Retourne la chaine traduite si possible.
     *
     * @param message message formate avec la meme syntaxe que {@link
     *                String#format}
     * @param args    les parametres pour le message.
     * @return la traduction si possible ou la chaine passee en parametre
     *         sinon.
     */
    public static String _(String message, Object... args) {

        // if the key to translate is null, just return null
        if (message == null) {
            return null;
        }

        I18nLanguage language = getCurrentLanguage();
        String result = message;
        if (language != null) {
            result = language.translate(message);
        }
        try {
            return applyFilter(String.format(result, args));
        } catch (Exception eee) {
            try {
                return applyFilter(String.format(message, args));
            } catch (Exception zzz) {
                if (log.isWarnEnabled()) {
                    log.warn(
                            _("nuitonutil.error.i18n.untranslated.message", message),
                            zzz);
                }
                return applyFilter(message);
            }
        }
    }

    /**
     * Retourne la chaine passée en argument.
     * <p/>
     * Utile surtout pour collecter les chaines et ne pas les traduires à leur
     * apparition.
     * <p/>
     * Par exemple :
     * <pre>String key = "nuitonutils.key";
     * String result = _(key)</pre>
     * fonctionnera, mais la chaine n'aura pas été marquée comme devant être
     * internationalisé.
     * <p/>
     * Tres utile par exemple, pour crée des objets non internationnalisé, et
     * devant être traduit seulement à leur lecture suivant la locale du lecteur
     * et non du créateur.
     *
     * @param message message formate avec la meme syntaxe que {@link
     *                String#format(String, Object...)}
     * @param args    les parametres pour le message.
     * @return le message passe en argument mais formatté avec les parametres
     */
    public static String n_(String message, Object... args) {
        try {
            return String.format(message, args);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn(
                        _("nuitonutil.error.i18n.unformated.message", message, Arrays.toString(args)),
                        eee);
            }
            return message;
        }
    }

    /**
     * Retourne la chaine passé en argument.
     * <p/>
     * Utile surtout pour collecter les chaines et ne pas les traduires à leur
     * apparition.
     * <p/>
     * Par exemple :
     * <pre>String key = "nuitonutils.key";
     * String result = _(key)</pre>
     * fonctionnera, mais la chaine n'aura pas été marquée comme devant être
     * internationalisé.
     * <p/>
     * Tres utile par exemple, pour crée des objets non internationnalisé, et
     * devant être traduit seulement à leur lecture suivant la locale du lecteur
     * et non du créateur.
     *
     * @param message la chaine à traduire
     * @return la chaine passée en argument.
     */
    public static String n_(String message) {
        return message;
    }

    /**
     * close i18n caches, says the store if exists
     * <p/>
     * This method should be called to reset all caches (languages,
     * bundles,...)
     */
    public static void close() {
        if (store != null) {
            store.close();
            store = null;
        }
    }

    public static I18nInitializer getInitializer() {
        return initializer;
    }

    /**
     * Get the i18n store.
     * <p/>
     * If store is not init, then instanciate it.
     *
     * @return the instanciated i18n store
     */
    public static I18nStore getStore() {

        if (store == null) {
            store = new I18nStore(DEFAULT_LOCALE, initializer);
        }
        return store;
    }

    /**
     * Applique le filtre s'il y en a un
     *
     * @param message le message qui devrait etre retourne avant application du
     *                filtre.
     * @return le message filtre
     */
    protected static String applyFilter(String message) {
        if (getFilter() != null) {
            return getFilter().applyFilter(message);
        }
        return message;
    }

    /**
     * @return the current language of the store, or {@code null} if store is
     *         not init.
     */
    protected static I18nLanguage getCurrentLanguage() {
        I18nLanguage language = store == null ? null : store.getLanguage();
        return language;
    }

    protected static I18nFilter getFilter() {
        return filter;
    }

    /**
     * @return the current store
     * @deprecated since 1.1, only keep for compatibility
     */
    @Deprecated
    public static I18nStore getLoader() {
        return getStore();
    }

    /**
     * Parse a list of {@link Locale} seperated by comma.
     * <p/>
     * Example : fr_FR,en_GB
     *
     * @param str the string representation of locale separated by comma
     * @return list of available locales
     * @throws IllegalArgumentException ia a locale is not valid
     * @deprecated since 1.1, prefer use the {@link I18nUtil#parseLocales(String)}
     */
    @Deprecated
    public static Locale[] parseLocales(String str)
            throws IllegalArgumentException {
        return I18nUtil.parseLocales(str);
    }

    /**
     * @param str the text representation of the locale
     * @return the locale, or default locale if could not parse the given one
     * @deprecated since 1.1, prefer use the {@link I18nUtil#newLocale(String)}
     */
    @Deprecated
    public static Locale newLocale(String str) {
        return I18nUtil.newLocale(str);
    }

    /**
     * @param language the language of the locale
     * @param country  the country of the locale
     * @return the required locale
     * @deprecated since 1.1, prefer use the {@link I18nUtil#newLocale(String,String)}
     */
    @Deprecated
    public static Locale newLocale(String language, String country) {
        return I18nUtil.newLocale(language, country);
    }
}
