/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserJavaMojo.java 2023 2014-02-04 05:46:16Z tchemit $
 * $HeadURL: https://svn.nuiton.org/i18n/tags/i18n-3.1/i18n-maven-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserJavaMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.parser.impl;

import org.antlr.v4.runtime.ANTLRInputStream;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.TokenStream;
import org.antlr.v4.runtime.misc.NotNull;
import org.antlr.v4.runtime.tree.ParseTree;
import org.apache.commons.io.IOUtils;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.i18n.plugin.parser.AbstractFileParser;
import org.nuiton.i18n.plugin.parser.AbstractI18nParserMojo;
import org.nuiton.i18n.plugin.parser.FileParser;
import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.i18n.plugin.parser.java.JavaBaseVisitor;
import org.nuiton.i18n.plugin.parser.java.JavaLexer;
import org.nuiton.i18n.plugin.parser.java.JavaParser;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.FileUpdaterHelper;
import org.nuiton.io.SortedProperties;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * To parse java files to detect new i18n keys.
 * <p/>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author jruchaud <ruchaud@codelutin.com>
 * @author tchemit <tchemit@codelutin.com>
 */
@Mojo(name = "parserJava", defaultPhase = LifecyclePhase.GENERATE_RESOURCES)
public class ParserJavaMojo extends AbstractI18nParserMojo {

    public static final String DEFAULT_INCLUDES = "**/*.java";

    /** Root directory of the default entry. */
    @Parameter(property = "i18n.defaultBasedir", defaultValue = "${basedir}/src/main/java")
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     * <p/>
     * <strong>Note:</strong> default value is **\/*.java
     */
    @Parameter(property = "i18n.defaultIncludes", defaultValue = DEFAULT_INCLUDES, required = true)
    protected String defaultIncludes;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @since 2.0
     */
    @Parameter(property = "i18n.outputGetter", defaultValue = "java.getter")
    protected String outputGetter;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return FileUpdaterHelper.newJavaFileUpdater(entry.getBasedir(), cp);
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    public FileParser newFileParser(Pattern acceptPattern) {

        return new JavaFileParser(getLog(),
                                  encoding,
                                  oldParser,
                                  acceptPattern,
                                  isShowTouchedFiles()
        );
    }

    protected static class JavaFileParser extends AbstractFileParser {

        public JavaFileParser(Log log,
                              String encoding,
                              SortedProperties oldParser,
                              Pattern acceptKeyPattern,
                              boolean showTouchedFiles) {
            super(log, encoding, oldParser, acceptKeyPattern, showTouchedFiles);
        }

        @Override
        public void parseFile(File file) throws IOException {

            Reader inputStream = new InputStreamReader(new FileInputStream(file), "UTF-8");
            try {
                // parse the file
                TokenStream tokenStream = new CommonTokenStream(new JavaLexer(new ANTLRInputStream(inputStream)));
                JavaParser parser = new JavaParser(tokenStream);

                JavaParser.CompilationUnitContext compilationUnitContext = parser.compilationUnit();
                compilationUnitContext.accept(new JavaParserVisitor(file));
                inputStream.close();
            } catch (Exception e) {
                throw new ParserException(e);
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        }

        @Override
        public void parseLine(File file, String line) throws IOException {
        }

        protected class JavaParserVisitor extends JavaBaseVisitor<Void> {

            protected final Set<String> simpleI18nMethodPrefix;

            protected final Set<String> complexI18nMethodPrefix;

            protected final File file;

            private JavaParserVisitor(File file) {
                this.file = file;
                simpleI18nMethodPrefix = new HashSet<String>();
                complexI18nMethodPrefix = new HashSet<String>();
                simpleI18nMethodPrefix.add("org.nuiton.i18n.I18n.n");
                simpleI18nMethodPrefix.add("org.nuiton.i18n.I18n.t");
                simpleI18nMethodPrefix.add("I18n.n");
                simpleI18nMethodPrefix.add("I18n.t");
                simpleI18nMethodPrefix.add("n");
                simpleI18nMethodPrefix.add("t");
                complexI18nMethodPrefix.add("org.nuiton.i18n.I18n.l");
                complexI18nMethodPrefix.add("I18n.l");
                complexI18nMethodPrefix.add("l");
            }

            @Override
            public Void visitStatementExpression(@NotNull JavaParser.StatementExpressionContext ctx) {
                return super.visitStatementExpression(ctx);
            }

            @Override
            public Void visitExpression(@NotNull JavaParser.ExpressionContext ctx) {

                boolean match = false;
                if (ctx.getChildCount() > 2 && ctx.getChild(2) instanceof JavaParser.ExpressionListContext) {
                    ParseTree child = ctx.getChild(0);
                    String childText = child.getText();
                    if (simpleI18nMethodPrefix.contains(childText)) {

                        // key is argument 1 of child 2
                        JavaParser.ExpressionListContext arguments = (JavaParser.ExpressionListContext) ctx.getChild(2);
                        String firstArgs = arguments.getChild(0).getText();
                        if (firstArgs.matches("^\"[^\"]+\"$")) {
                            String key = firstArgs.substring(1).substring(0, firstArgs.length() - 2);
                            if (getLog().isDebugEnabled()) {
                                getLog().debug(file.getName() + " detected key = " + key);
                            }
                            ParserJavaMojo.JavaFileParser.this.registerKey(key);
                        }
                        match = true;
                    } else if (complexI18nMethodPrefix.contains(childText)) {

                        // key is argument 2 of child 2
                        JavaParser.ExpressionListContext arguments = (JavaParser.ExpressionListContext) ctx.getChild(2);
                        String firstArgs = arguments.getChild(2).getText();
                        if (firstArgs.matches("^\"[^\"]+\"$")) {
                            String key = firstArgs.substring(1).substring(0, firstArgs.length() - 2);
                            if (getLog().isDebugEnabled()) {
                                getLog().debug(file.getName() + " detected key = " + key);
                            }
                            ParserJavaMojo.JavaFileParser.this.registerKey(key);
                        }
                        match = true;
                    }
                }
                Void aVoid = null;
                if (!match) {
                    // continue visit
                    aVoid = super.visitExpression(ctx);
                }
                return aVoid;
            }
        }
    }


}
