package org.nuiton.eugene.plugin.renderer;

/*
 * #%L
 * Nuiton Utils :: Maven Report Plugin
 * $Id: AbstractEugeneReportRenderer.java 1360 2014-06-01 11:17:25Z tchemit $
 * $HeadURL: https://nuiton.org/svn/nuiton-maven-report-plugin/trunk/src/main/java/org/nuiton/config/ApplicationConfigReportRenderer.java $
 * %%
 * Copyright (C) 2012 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.collections4.MapUtils;
import org.apache.maven.doxia.sink.Sink;
import org.apache.maven.doxia.sink.SinkEventAttributeSet;
import org.apache.maven.doxia.sink.SinkEventAttributes;
import org.apache.maven.reporting.AbstractMavenReportRenderer;
import org.codehaus.plexus.i18n.I18N;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelEnumeration;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;

import java.util.Locale;
import java.util.Map;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 2.10
 */
public abstract class AbstractEugeneReportRenderer<D, DD> extends AbstractMavenReportRenderer {

    /**
     * Internationalization component.
     */
    protected final I18N i18n;

    /**
     * The locale we are rendering for.
     */
    protected final Locale locale;

    /**
     * The name of the bundle containing our I18n resources.
     */
    protected final String bundleName;

    /**
     * Data to describe.
     */
    protected final D data;

    protected final String javaDocDestDir;

    protected final String jxrDestDir;

    protected abstract Map<String, DD> getDataDetails();

    public AbstractEugeneReportRenderer(Sink sink,
                                        I18N i18n,
                                        Locale locale,
                                        String bundleName,
                                        String javaDocDestDir,
                                        String jxrDestDir,
                                        D data) {
        super(sink);
        this.i18n = i18n;
        this.locale = locale;
        this.bundleName = bundleName;
        this.javaDocDestDir = javaDocDestDir;
        this.jxrDestDir = jxrDestDir;
        this.data = data;
        this.sink = sink;
    }

    @Override
    public String getTitle() {
        return getText("report.title");
    }

    @Override
    public void renderBody() {

        String implementation = data.getClass().getName();

        sink.section1();

        sink.sectionTitle1();
        sink.text(getText("report.overview.title"));
        sink.sectionTitle1_();

        sink.lineBreak();

        sink.paragraph();
        sink.text(getText("report.overview.text"));
        sink.text(" ");
        sink.bold();
        sink.text(implementation);
        sink.bold_();
        sink.text(" ");
        sink.link("./" + javaDocDestDir + "/" + implementation.replace('.', '/') + ".html");
        sink.text("javadoc");
        sink.link_();
        sink.link("./" + jxrDestDir + "/" + implementation.replace('.', '/') + ".html");
        sink.text(" ");
        sink.text("xref");
        sink.link_();
        sink.text(".");
        sink.paragraph_();

        sink.paragraph();
        sink.link(getEugeneDocLink());
        sink.text(getText("report.overview.more.information"));
        sink.link_();
        sink.paragraph_();

        Map<String, DD> detailEntries = getDataDetails();

        renderSummaryTable(data, detailEntries);

        sink.section1_();


        if (MapUtils.isNotEmpty(detailEntries)) {

            sink.section1();

            sink.sectionTitle1();
            sink.text(getText("report.detail.title"));
            sink.sectionTitle1_();

            sink.lineBreak();

            sink.paragraph();
            sink.text(getText("report.detail.text"));
            sink.paragraph_();

            for (Map.Entry<String, DD> detailEntry : detailEntries.entrySet()) {

                renderDetail(detailEntry.getKey(), detailEntry.getValue());

            }

            sink.section1_();

        }

    }

    protected abstract String getEugeneDocLink();

    protected abstract void renderSummaryTable(D provider, Map<String, DD> detailEntries);

    protected abstract void renderDetail(String key, DD detail);

    protected String getText(boolean key) {
        return getText("report." + String.valueOf(key));
    }

    /**
     * Gets the localized message for this report.
     *
     * @param key the message key.
     * @return the message.
     */
    protected String getText(String key) {
        return i18n.getString(bundleName, locale, key);
    }

    protected void renderWarningIcon() {
        sink.figure();
        sink.figureGraphics("images/icon_warning_sml.gif");
        sink.figure_();
    }

    protected void renderErrorIcon() {
        sink.figure();
        sink.figureGraphics("images/icon_error_sml.gif");
        sink.figure_();
    }

    protected void renderSuccessIcon() {
        sink.figure();
        sink.figureGraphics("images/icon_success_sml.gif");
        sink.figure_();
    }

    protected void renderInfoIcon() {
        sink.figure();
        sink.figureGraphics("images/icon_info_sml.gif");
        sink.figure_();
    }

    protected void sinkHeaderCellText(String text) {
        sink.tableHeaderCell();
        sink.text(text);
        sink.tableHeaderCell_();
    }

    protected void sinkHeaderCellText(SinkEventAttributes width, String text) {
        sink.tableHeaderCell(width);
        sink.text(text);
        sink.tableHeaderCell_();
    }

    protected void sinkCellText(SinkEventAttributes width, String text) {
        sink.tableCell(width);
        sink.text(text);
        sink.tableCell_();
    }

    protected void sinkCellText(String text) {
        sink.tableCell();
        sink.text(text);
        sink.tableCell_();
    }

    protected void sinkCellVerbatimText(String text) {
        sink.tableCell();
        sink.verbatim(SinkEventAttributeSet.MONOSPACED);
        sink.text(text);
        sink.verbatim_();
        sink.tableCell_();
    }

    protected void sinkCellLink(String text, String url) {
        sink.tableCell();
        sink.link(url);
        sink.text(text);
        sink.link_();
        sink.tableCell_();
    }

    protected void renderTargets(Class<?>... targets) {

        boolean moreThanOne = targets.length > 1;

        if (moreThanOne) {

            sink.list_();

        }

        for (Class<?> target : targets) {

            if (moreThanOne) {

                sink.listItem();

            }

            renderDetailTarget(target);

            if (moreThanOne) {

                sink.listItem_();

            }

        }

        if (moreThanOne) {

            sink.list_();

        }

    }

    protected void renderDetailTarget(Class<?> target) {

        if (ObjectModel.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.model"));

        } else if (ObjectModelClassifier.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.classifier"));

        } else if (ObjectModelClass.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.class"));

        } else if (ObjectModelInterface.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.interface"));

        } else if (ObjectModelEnumeration.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.enumeration"));

        } else if (ObjectModelAttribute.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.attribute"));

        } else if (ObjectModelParameter.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.parameter"));

        } else if (ObjectModelOperation.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.operation"));

        } else if (ObjectModelElement.class.isAssignableFrom(target)) {

            sink.text(getText("report.detail.target.element"));

        }

    }

}
