/*
 * #%L
 * JMexico :: Model
 * 
 * $Id: InputDesignParser.java 65 2012-04-30 12:06:54Z sletellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/jmexico/tags/jmexico-0.5/jmexico-model/src/main/java/fr/reseaumexico/model/parser/InputDesignParser.java $
 * %%
 * Copyright (C) 2011 Réseau Mexico, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package fr.reseaumexico.model.parser;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.reseaumexico.model.ExperimentDesign;
import fr.reseaumexico.model.Factor;
import fr.reseaumexico.model.InputDesign;
import fr.reseaumexico.model.InputDesignImpl;
import fr.reseaumexico.model.MexicoTechnicalException;
import fr.reseaumexico.model.Scenario;
import fr.reseaumexico.model.ScenarioImpl;
import fr.reseaumexico.model.ValueType;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.List;
import java.util.Map;

import static fr.reseaumexico.model.MexicoXmlConstant.FACTOR;
import static fr.reseaumexico.model.MexicoXmlConstant.INPUT_DESIGN;
import static fr.reseaumexico.model.MexicoXmlConstant.INPUT_DESIGN_DATE;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_FACTOR_VALUES;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_NAME;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_ORDER_NUMBER;

/**
 * Parser for input design xml files, build a {@link InputDesign} model
 *
 * @author sletellier <letellier@codelutin.com>
 * @since 0.1
 */
public class InputDesignParser extends MexicoXmlParser<InputDesign> {

    protected ExperimentDesignParser experimentDesignParser;

    public InputDesignParser(File file) throws FileNotFoundException {
        super(file);
        experimentDesignParser = new ExperimentDesignParser(file);
    }

    @Override
    protected InputDesign parseModel(XmlPullParser parser) throws IOException, XmlPullParserException, ParseException {
        InputDesign model = new InputDesignImpl();

        parseInputDesignMeta(parser, model);

        // parse experiment design
        ExperimentDesign experimentDesign = experimentDesignParser.parseModel(parser);
        model.setExperimentDesign(experimentDesign);

        parseScenarios(parser, model);

        return model;
    }

    protected void parseInputDesignMeta(XmlPullParser parser, InputDesign model) throws IOException, XmlPullParserException, ParseException {

        // file must start with input design tag
        if (parser.next() == XmlPullParser.START_TAG &&
                !parserEqual(parser, INPUT_DESIGN)) {
            throw new MexicoTechnicalException("Input design file must start with " + INPUT_DESIGN + " tag");
        }

        // parse date
        String dateAsString = parser.getAttributeValue(null, INPUT_DESIGN_DATE);
        model.setDate(parseDate(dateAsString));
    }

    protected void parseScenarios(XmlPullParser parser, InputDesign model) throws IOException, XmlPullParserException {

        // scenarios
        if (parser.nextTag() == XmlPullParser.START_TAG &&
                parserEqual(parser, SCENARIO)) {

            List<Scenario> scenarios = Lists.newArrayList();
            while (!(parser.getEventType() == XmlPullParser.END_TAG &&
                    parserEqual(parser, INPUT_DESIGN))) {

                // factor
                scenarios.add(parseScenario(parser));
            }
            model.setScenario(scenarios);
        }
    }

    protected Scenario parseScenario(XmlPullParser parser) throws IOException, XmlPullParserException {
        Scenario scenario = new ScenarioImpl();

        // name
        String name = parser.getAttributeValue(null, SCENARIO_NAME);
        scenario.setName(name);

        // orderValue
        String orderValue = parser.getAttributeValue(null, SCENARIO_ORDER_NUMBER);
        scenario.setOrderNumber(Integer.parseInt(orderValue));

        // factor values
        if (parser.nextTag() == XmlPullParser.START_TAG &&
                parserEqual(parser, SCENARIO_FACTOR_VALUES)) {

            Map<Factor, Object> factorValues = Maps.newLinkedHashMap();
            while (!(parser.getEventType() == XmlPullParser.END_TAG &&
                    parserEqual(parser, SCENARIO))) {

                // get factor
                String factorId = parser.getAttributeValue(null, FACTOR);
                Factor factor = experimentDesignParser.getFactors().get(factorId);

                ValueType valueType = factor.getDomain().getValueType();

                // factor value
                String value = parser.nextText();

                factorValues.put(factor, getTypedValue(valueType, value));

                // read next tag
                parser.nextTag();
            }
            scenario.setFactorValues(factorValues);
        }

        // read next tag
        parser.nextTag();

        return scenario;
    }
}
