/*
 * #%L
 * IsisFish
 * 
 * $Id: ParameterTableCellEditor.java 4156 2014-12-09 11:27:18Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/ui/widget/editor/ParameterTableCellEditor.java $
 * %%
 * Copyright (C) 2006 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget.editor;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractCellEditor;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.table.TableCellEditor;

import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.TimeStep;
import fr.ifremer.isisfish.ui.widget.editor.FactorEditorListener.FactorCallback;
import fr.ifremer.isisfish.util.ConverterUtil;

/**
 * Un editeur generic de champs qui se base sur le model de données
 * a editer.
 *
 * Created: 25 sept. 06 12:35:21
 *
 * @author poussin
 * @author chemit
 * @version $Revision: 4156 $
 *
 * Last update: $Date: 2014-12-09 12:27:18 +0100 (mar., 09 déc. 2014) $
 * by : $Author: echatellier $
 */
public class ParameterTableCellEditor extends AbstractCellEditor implements TableCellEditor {

    private static final long serialVersionUID = 6860330126841984303L;

    /** Class logger. */
    private static Log log = LogFactory.getLog(ParameterTableCellEditor.class);

    protected RegionStorage regionStorage = null;

    protected Component editor = null;

    protected Class<?> type = null;

    protected FactorEditorListener factorActionListener;

    private static enum TypeClassMapping {
        /** for simple int. */
        Int(int.class, Integer.class),
        /** for a topia entity. */
        Topia(TopiaEntity.class),
        /** for a mounth. */
        Mounth(Month.class),
        /** for a simple date. */
        TimeStep(TimeStep.class),
        /** for a simple boolean. */
        Double(double.class, Double.class),
        /** for a simple boolean. */
        Boolean(boolean.class, Boolean.class),
        /** for a Factor. */
        Factor(Factor.class),
        /** for a file location. */
        File(java.io.File.class),
        /** for a simple String. */
        String(java.lang.String.class);

        private final Class<?>[] klazz;

        private TypeClassMapping(Class<?>... klazz) {
            this.klazz = klazz;
        }

        protected static TypeClassMapping getMapping(Class<?> klazz) {
            for (TypeClassMapping t : TypeClassMapping.values()) {
                for (Class<?> loopKlazz : t.klazz) {
                    if (loopKlazz.isAssignableFrom(klazz)) {
                        return t;
                    }
                }
            }
            throw new RuntimeException(
                    "could not found a TypeClassMapping for this class "
                            + klazz);
        }
    }

    /**
     * Return region storage.
     * 
     * @return returns the region storage.
     */
    public RegionStorage getRegion() {
        return regionStorage;
    }

    /**
     * Set region storage.
     * 
     * @param regionStorage the region storage to set.
     */
    public void setRegionStorage(RegionStorage regionStorage) {
        this.regionStorage = regionStorage;
        if (regionStorage == null && log.isWarnEnabled()) {
            log.warn("setRegionStorage called with null value in parameters table cell editor");
        }
    }

    public void setFactorActionListener(FactorEditorListener factorActionListener) {
        this.factorActionListener = factorActionListener;
    }

    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int column) {

        // get value type
        type = getType(value, row, column);
        JComboBox c;
        //if (!table.hasFocus()) return null;

        TypeClassMapping mapping = TypeClassMapping.getMapping(type);
        switch (mapping) {
        case Topia:
            // on a un bean comme parametre
            try {
                if (regionStorage == null) {
                    return null;
                }
                TopiaContext context = regionStorage.getStorage().beginTransaction();
                if (context != null) {
                    List list = context.findAll("from " + type.getName());
                    c = new JComboBox(list.toArray());
                    c.setSelectedItem(value);
                    editor = c;
                    context.closeContext();
                }

            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't get entity object for combobox", eee);
                }
            }
            break;
        case Boolean:
            List<Boolean> listB = new ArrayList<Boolean>();
            listB.add(Boolean.TRUE);
            listB.add(Boolean.FALSE);
            c = new JComboBox(listB.toArray());
            listB.clear();
            // TODO See how to convert
            c.setSelectedItem(value);
            editor = c;
            break;
        case TimeStep:
            TimeStep date = (TimeStep) value;
            if (date == null) {
                date = new TimeStep(0);
            }
            int month = date.getMonth().getMonthNumber();
            int yea = date.getYear();
            editor = new StepComponent(month, yea);
            //editor = new JTextField(String.valueOf(date.getDate()));
            break;
        case Mounth:
            editor = MonthComponent.createMounthCombo(((Month) value).getMonthNumber());
            break;
        case Factor:
            editor = new FactorEditor(this, (Factor)value);
            if (factorActionListener == null) {
                throw new RuntimeException("Missing factorActionListener call setFactorActionListener()");
            }
            factorActionListener.setFactorCallback((FactorCallback)editor);
            ((JButton)editor).addActionListener(factorActionListener);
            break;
        case File:
            // break;
        default:
            editor = new JTextField(String.valueOf(value));
        }
        if (editor != null) {
            if (JComboBox.class.isInstance(editor)) {
                ((JComboBox) editor).addActionListener(getStopEditingListener());
            } else if (JTextField.class.isInstance(editor)) {
                ((JTextField) editor).addFocusListener(new FocusAdapter() {
                    @Override
                    public void focusLost(FocusEvent e) {
                        if (log.isDebugEnabled()) {
                            log.debug("Stop cell editing");
                        }
                        stopCellEditing();
                    }
                });
            } else if (StepComponent.class.isInstance(editor)) {
                StepComponent date = ((StepComponent) editor);
                if (date.getMounthCombo() != null) {
                    date.getMounthCombo().addActionListener(getStopEditingListener());
                }
                if (date.getYearCombo() != null) {
                    date.getYearCombo().addActionListener(getStopEditingListener());
                }
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("getCellEditorValue [" + type + "] [mapping:" + mapping + "="
                + editor);
        }

        return editor;
    }

    /**
     * Get type for value.
     * 
     * Must be overridden (value can be null).
     * 
     * @param value
     * @param row
     * @param column
     * @return type for value
     */
    protected Class<?> getType(Object value, int row, int column) {
        // default to value's class
        return value.getClass();
    }

    protected ActionListener getStopEditingListener() {
        return new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                stopCellEditing();
            }
        };
    }

    public Object getCellEditorValue() {

        Object result = null;
        TopiaContext context = null;
        if (editor == null) {
            return "";
        }
        TypeClassMapping mapping = TypeClassMapping.getMapping(type);
        switch (mapping) {
        case Topia:
            result = ((JComboBox) editor).getSelectedItem();
            break;
        case Boolean:
            result = ((JComboBox) editor).getSelectedItem();
            break;
        case Mounth:
            result = new Month(((MonthComponent)editor).getSelectedValue());
            break;
        case TimeStep:
            result = new TimeStep(((StepComponent) editor).getSelectedValue());
            break;
        case Factor:
            result = ((FactorEditor)editor).getFactor();
            break;
        case File:
            //TODO
            // result = getTextFieldValue(editor);break;
        default:
            if (editor instanceof JTextField) {
                result = ((JTextField) editor).getText();
            } else if (editor instanceof JComboBox) {
                result = ((JComboBox) editor).getSelectedItem();
            }
        }

        if (result instanceof String && !String.class.equals(type)) {
            if (regionStorage != null) {
                context = regionStorage.getStorage();
            }
            ConvertUtilsBean cub = ConverterUtil.getConverter(context);
            result = cub.convert((String) result, type);
        }

        if (log.isDebugEnabled()) {
            log.debug("getCellEditorValue [" + type + "] [mapping:" + mapping
                + "]= " + result);
        }

        return result;
    }
}
