/*
 * #%L
 * IsisFish
 * 
 * $Id: LogConsoleModel.java 4156 2014-12-09 11:27:18Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/logging/console/LogConsoleModel.java $
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging.console;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * This class is the model used in LogConsole.
 * 
 * The model deals with ChangeEvent to notify ui, model has changed.
 *
 * @author chemit
 */
public class LogConsoleModel {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    //static private Log log = LogFactory.getLog(LogConsoleModel.class);

    /** the maximum number of items to display in ui */
    protected int nbLinesInEditor;

    /** the size of the stream */
    protected long nbLines;

    /** the text to filter */
    protected String searchText;
    /** the levels to filter */
    protected int levels;

    /** the real log file */
    protected File logFile;
    /** server to use for mail send */
    protected String smtpServer;
    /** default destinataire for mail send */
    protected String to;
    /** default from from mail send */
    protected String from;

    /** the position of the first line in model */
    protected long firstLinePosition;

    /** list of all items we keep in store */
    protected List<String> allItems;

    /** to contains listeners */
    protected EventListenerList listenerList = new EventListenerList();

    /** The unique changeEvent */
    protected ChangeEvent changeEvent;

    public LogConsoleModel(File logFile, int nbLinesInEditor, String smtpServer, String fromMail, String to) {
        this.nbLinesInEditor = nbLinesInEditor;
        this.listenerList = new EventListenerList();
        this.allItems = new ArrayList<String>();

        this.logFile = logFile;
        this.smtpServer = smtpServer;
        this.from = fromMail;
        this.to = to;

        // by default use all levels
        this.levels = 0;

        // by default no text to filter
        this.searchText = "";
    }

    /** @return the text to be displayed in ui */
    public String getText() {
        StringBuffer sb = new StringBuffer();
        if (!allItems.isEmpty()) {
            for (String item : allItems) {
                sb.append('\n').append(item);
            }
        } else {
            sb.append(" no log found...");
        }
        return sb.toString().substring(1);
    }

    public boolean canReset() {
        return levels > 0 || !"".equals(searchText);
    }

    public long getNbLines() {
        return nbLines;
    }


    public File getLogFile() {
        return logFile;
    }

    public long getFirstLinePosition() {
        if (allItems.isEmpty()) {
            return -1;
        }
        return firstLinePosition;
    }

    public long getLastLinePosition() {
        if (allItems.isEmpty()) {
            return -1;
        }
        return firstLinePosition + allItems.size();
    }

    public int getLevels() {
        return levels;
    }

    public String getSearchText() {
        return searchText;
    }

    public void setSmtpServer(String smtpServer) {
        this.smtpServer = smtpServer;
    }

    public void setEmailTo(String to) {
        this.to = to;
    }

    public void setEmailFrom(String from) {
        this.from = from;
    }

    public void setFirstLinePosition(long firstLinePosition) {
        this.firstLinePosition = firstLinePosition;
    }

    public void setSearchText(String searchText) {
        this.searchText = searchText;
    }

    public void setNbLines(long nbLines) {
        this.nbLines = nbLines;
    }

    public void setLevels(int levels) {
        this.levels = levels;
    }

    public int getNbLinesInEditor() {
        return nbLinesInEditor;
    }
    
    public void setNbLinesInEditor(int nbLinesInEditor) {
        this.nbLinesInEditor = nbLinesInEditor;
    }

    public boolean isBOF() {
        return firstLinePosition == 0;
    }

    public boolean isEOF() {
        return getLastLinePosition() == nbLines;
    }

    public String getFrom() {
        return from;
    }

    public String getSmtpServer() {
        return smtpServer;
    }

    /**
     * Adds a <code>ChangeListener</code> to the button.
     *
     * @param l the listener to be added
     */
    public void addChangeListener(ChangeListener l) {
        listenerList.add(ChangeListener.class, l);
    }

    /**
     * Removes a ChangeListener from the button.
     *
     * @param l the listener to be removed
     */
    public void removeChangeListener(ChangeListener l) {
        listenerList.remove(ChangeListener.class, l);
    }

    /**
     * Returns an array of all the <code>ChangeListener</code>s added
     * to this AbstractButton with addChangeListener().
     *
     * @return all of the <code>ChangeListener</code>s added or an empty
     *         array if no listeners have been added
     * @since 1.4
     */
    public ChangeListener[] getChangeListeners() {
        return listenerList.getListeners(ChangeListener.class);
    }

    /**
     * Notifies all listeners that have registered interest for
     * notification on this event type.  The event instance
     * is lazily created.
     *
     * @see EventListenerList
     */
    protected void fireStateChanged() {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == ChangeListener.class) {
                // Lazily create the event:
                if (changeEvent == null) {
                    changeEvent = new ChangeEvent(this);
                }
                ((ChangeListener) listeners[i + 1]).stateChanged(changeEvent);
            }
        }
    }
}
