package com.formos.tapestry.testify.internal;

import static java.lang.String.format;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;

import org.apache.tapestry5.ioc.ObjectCreator;
import org.apache.tapestry5.ioc.ServiceLifecycle;
import org.apache.tapestry5.ioc.ServiceResources;
import org.apache.tapestry5.ioc.services.Builtin;
import org.apache.tapestry5.ioc.services.ClassFab;
import org.apache.tapestry5.ioc.services.ClassFactory;
import org.apache.tapestry5.ioc.services.MethodSignature;

/**
 * Allows a service to exist "per test" (on the assumption that a test runs in only one thread).
 * This involves an inner proxy, which caches an object derived from a
 * {@link org.apache.tapestry5.ioc.ObjectCreator} as a key in the {@link PerTestDataStore}.
 * Method invocations are delegated to the per-test service instance.
 * <p/>
 * This scheme ensures that, although the service builder method will be invoked many times
 * over the life of the application, the service decoration process occurs only once.
 * The final calling chain is: Service Proxy --&gt; Decorator(s) --&gt; PerTest Proxy --&gt; (per test) instance.
 */
public class PerTestServiceLifecycle implements ServiceLifecycle {
    private static final String PER_TEST_METHOD_NAME = "_perTestInstance";

    private final PerTestDataStore perTestManager;
    private final ClassFactory classFactory;


    public PerTestServiceLifecycle(@Builtin PerTestDataStore perTestManager, @Builtin ClassFactory classFactory) {
        this.perTestManager = perTestManager;
        this.classFactory = classFactory;
    }


    /**
     * Returns false; this lifecycle represents a service that will be created many times (by each thread).
     */
    public boolean isSingleton() {
        return false;
    }


    public Object createService(ServiceResources resources, ObjectCreator creator) {
        Class<?> proxyClass = createProxyClass(resources);

        ObjectCreator perTestCreator = new PerTestServiceCreator(perTestManager, creator);

        try {
            Constructor<?> ctor = proxyClass.getConstructors()[0];
            return ctor.newInstance(perTestCreator);
        } catch (InvocationTargetException ex) {
            throw new RuntimeException(ex.getCause());
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }


    private Class<?> createProxyClass(ServiceResources resources) {
        Class<?> serviceInterface = resources.getServiceInterface();
        ClassFab cf = classFactory.newClass(serviceInterface);
        cf.addField("_creator", Modifier.PRIVATE | Modifier.FINAL, ObjectCreator.class);

        // Constructor takes a ServiceCreator
        cf.addConstructor(new Class[] { ObjectCreator.class }, null, "_creator = $1;");

        String body = format("return (%s) _creator.createObject();", serviceInterface.getName());
        MethodSignature sig = new MethodSignature(serviceInterface, PER_TEST_METHOD_NAME, null, null);
        cf.addMethod(Modifier.PRIVATE, sig, body);

        String toString = format("<PerTest Proxy for %s(%s)>", resources.getServiceId(), serviceInterface.getName());
        cf.proxyMethodsToDelegate(serviceInterface, PER_TEST_METHOD_NAME + "()", toString);

        return cf.createClass();
    }
}
