package com.formos.tapestry.testify.core;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

import org.apache.tapestry5.annotations.Service;
import org.apache.tapestry5.internal.test.TestableRequest;
import org.apache.tapestry5.internal.test.TestableResponse;
import org.apache.tapestry5.ioc.Registry;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.internal.services.AccessableObjectAnnotationProvider;
import org.apache.tapestry5.ioc.services.MasterObjectProvider;
import org.apache.tapestry5.services.RequestHandler;
import org.apache.tapestry5.services.Session;
import org.apache.tapestry5.test.PageTester;

import com.formos.tapestry.testify.internal.AccessObjectFromField;
import com.formos.tapestry.testify.internal.ObjectsForComponentsStore;
import com.formos.tapestry.testify.internal.PerTestDataStore;
import com.formos.tapestry.testify.internal.TestifyModule;

/**
 * An extension of Tapestry's PageTester to add additional features.
 */
public class TapestryTester extends PageTester {

    private final TestableRequest request;
    private final TestableResponse response;
    private final RequestHandler requestHandler;


    public TapestryTester(String appPackage, String appName, String contextPath, Class<?>... moduleClasses) {
        super(appPackage, appName, contextPath, addTestify(moduleClasses));
        
        Registry registry = getRegistry();
        request = registry.getService(TestableRequest.class);
        response = registry.getService(TestableResponse.class);
        requestHandler = registry.getService("RequestHandler", RequestHandler.class);
    }

    
    public TapestryTester(String appPackage, Class<?>... moduleClasses) {
		this(appPackage, "app", PageTester.DEFAULT_CONTEXT_PATH, moduleClasses);
	}

    
    private static Class<?>[] addTestify(Class<?>[] moduleClasses) {
        Class<?>[] result = new Class<?>[moduleClasses.length+1];
        result[0] = TestifyModule.class;
        System.arraycopy(moduleClasses, 0, result, 1, moduleClasses.length);
        return result;
    }



    /**
     * Obtains a service via its unique service id.
     *
     * @param <T>
     * @param id unique Service id used to locate the service object (may contain <em>symbols</em>, which
     *           will be expanded), case is ignored (not null)
     * @param serviceInterface the interface implemented by the service (or an interface extended by the service
     *                         interface) (not null)
     * @return the service found (not null)
     * @throws RuntimeException if the service is not defined, or if an error occurs instantiating it
     */
	public <T> T getService(String id, Class<T> serviceInterface) {
		assert id != null;
		assert serviceInterface != null;
        return getRegistry().getService(id, serviceInterface);
    }

	
    /**
     * Autobuilds a class by finding the public constructor with the most parameters. Services and resources will be
     * injected into the parameters of the constructor.
     *
     * @param <T>
     * @param clazz the type of object to instantiate (not null)
     * @return the instantiated instance (not null)
     * @throws RuntimeException if the autobuild fails
     * @see MasterObjectProvider
     */
    public <T> T autobuild(Class<T> clazz) {
	    return getRegistry().autobuild(clazz);
	}

    
	/**
	 * Call this method to mark the end of a test (for example in the test's tearDown() method). This
	 * is used to mark the end of the {@link TestifyConstants#PERTEST} scope.
	 */
    public void endTest() {
        getService(PerTestDataStore.class).cleanup();
        clearSession(); // Would be better if we could make the request per-test scope...
    }


	private void clearSession() {
        Session session = request.getSession(false);
        if (session != null) {
            for (String name : session.getAttributeNames()) {
                session.setAttribute(name, null);
            }
        }
    }


    /**
     * Process the {@link Inject} annotation on fields in the given object - those fields will be
     * given values found in this tester's IOC container. Use {@link Service} on the field as well
     * if you need to find a service by id.
     * 
     * @param object the object to process (not null)
     */
    public void injectInto(Object object) {
    	assert object != null;
        processFieldsAnnotatedWith(object, Inject.class, new FieldInjector());
    }

    
    /**
     * Searches for fields in the object annotated with {@link ForComponents} and registers them in
     * the component injection pipeline so that they will be the preferred objects to inject into
     * components and pages during this test. These objects are removed from the pipeline
     * when {@link #endTest()} is called.
     *  
     * @param object the object to process (not null)
     */
    public void collectForComponentsFrom(Object object) {
    	assert object != null;
        processFieldsAnnotatedWith(object, ForComponents.class, new TestObjectCollector());
    }

    
    /**
     * Processes a request for the specified path and returns the response generated. This is for
     * low-level testing, most of the time {@link #renderPage(String)} is what you want. Note that
     * unlike {@link #renderPage(String)}, this does not follow redirects - instead the redirect response
     * is returned.
     * 
     * @param path the path for the request; must begin with a "/" character (not null)
     * @return the response (not null)
     */
    public TestableResponse renderResponse(String path) {
        assert path.startsWith("/");
        try {
            request.clear().setPath(path);
            response.clear();

            boolean handled = requestHandler.service(request, response);
            if (!handled) {
                throw new RuntimeException(
                        String.format("Request was not handled: '%s' may not be a valid path.", path));
            }
            return response;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
    
    

    private void processFieldsAnnotatedWith(Object object, Class<? extends Annotation> annotation, FieldProcessor processor) {
        for (Class<?> asType = object.getClass(); !asType.equals(Object.class); asType = asType.getSuperclass()) {
            for (Field field : asType.getDeclaredFields()) {
                if (field.getAnnotation(annotation) != null) {
                    processField(object, processor, field);
                }
            }
        }
    }


    private void processField(Object object, FieldProcessor processor, Field field) {
        try {
            field.setAccessible(true);
            processor.process(object, field);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    
    private interface FieldProcessor {
        void process(Object object, Field field) throws IllegalAccessException;
    }

    
    private final class FieldInjector implements FieldProcessor {
        public void process(Object object, Field field) throws IllegalArgumentException, IllegalAccessException {
            Class<?> type = field.getType();
            Object value = getRegistry().getObject(type, new AccessableObjectAnnotationProvider(field));
            field.set(object, value);
        }
    }

    
    private class TestObjectCollector implements FieldProcessor {
        public void process(Object object, Field field) {
            AccessObjectFromField accessor = new AccessObjectFromField(object, field);
			String id = field.getAnnotation(ForComponents.class).value();
			getService(ObjectsForComponentsStore.class).put(accessor, id);
        }
    }

}
