package org.sharengo.wikitty.generator;

import java.io.IOException;
import java.io.Writer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.ImportsManager;
import org.nuiton.eugene.ObjectModelGenerator;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelDependency;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;

public class WikengoCommonGenerator extends ObjectModelGenerator {

    private static final Log log = LogFactory.getLog(WikengoCommonGenerator.class);

    protected ImportsManager imports;

    protected void clearImports() {
        if (imports == null) {
            imports = new ImportsManager();
        } else {
            imports.clearImports();
        }
    }

    protected void addImport(String fqn) {
        if (containsClassAndGeneric(fqn)) {
            String[] type = getClassAndGeneric(fqn);
            imports.addImport(checkForDatatype(type[0]));
            addImport(checkForDatatype(type[1]));
        } else if (isArray(fqn)) {
            imports.addImport(checkForDatatype(fqn.substring(0, fqn.length() - 2)));
        } else if (containsComma(fqn)) {
            String[] fqnCommaSeparated = splitByComma(fqn);
            for (String str : fqnCommaSeparated) {
                if (str != null && !str.isEmpty()) {
                    addImport(str.trim());
                }
            }
        } else {
            fqn = checkForDatatype(fqn);
            imports.addImport(fqn);
        }
    }

    private String[] splitByComma(String fqn) {
        return fqn.split(",");
    }

    private boolean containsComma(String fqn) {
        return fqn != null && fqn.indexOf(",") != -1;
    }

    protected void addImport(Class<?> clazz) {
        imports.addImport(clazz);
    }

    protected void addImport(ObjectModelClass clazz) {
        if (clazz != null) {
            addImport(clazz.getQualifiedName());
        }
    }

    /**
     * Return the minimum syntax for the type. The result depend of import added
     * by addImport.
     *
     * @param fqn the fully qualified name of type
     * @return minimum needed type
     */
    protected String getType(String fqn) {
        String result = getType(fqn, false);
        return result;
    }

    /**
     * Return the minimum syntax for the type. The result depend of import added
     * by addImport.
     *
     * @param fqn the fully qualified name of type
     * @param convert if true try to convert some type to other
     * (ex: enum to string, dto to string)
     * @return minimum needed type
     */
    protected String getType(String fqn, boolean convert) {
        // if type is Wikitty then we used String
        if ("org.sharengo.wikitty.Wikitty".equals(fqn)
                || "Wikitty".equals(fqn)) {
            fqn = "String";
        } else if (convert && null != getModel().getEnumeration(fqn)) {
            // if type is Wikitty then we used String
            fqn = "String";
        } else if (convert && getModel().hasClass(fqn)
                && EugengoUtils.isBusinessEntity(getModel().getClass(fqn))) {
            // for wikittyDto we use String for Id
            fqn = "String";
        }

        if (containsClassAndGeneric(fqn)) {
            String[] type = getClassAndGeneric(fqn);
            return imports.getType(checkForDatatype(type[0])) + "<" + getType(type[1], convert) + ">";
        } else if (containsComma(fqn)) {
            String result = "";
            for (String str : splitByComma(fqn)) {
                if (str != null && !str.isEmpty()) {
                    if (!result.isEmpty()) {
                        result += ", ";
                    }
                    result += str.trim();
                }
            }
            return result;
        } else {
            return imports.getType(checkForDatatype(fqn));
        }
    }

    protected boolean isArray(String fqn) {
        return fqn != null && fqn.trim().endsWith("[]");
    }

    protected boolean containsClassAndGeneric(String fqn) {
        return fqn != null && fqn.indexOf("<") != -1;
    }

    protected String[] getClassAndGeneric(String fqn) {
        int idx = fqn.indexOf("<");
        String[] result = new String[2];
        result[0] = fqn.substring(0, idx);
        result[1] = fqn.substring(idx+1, fqn.length() - 1);
        return result;
    }

    protected void generateImports(Writer output, String currentPackage) throws IOException {
        List<String> imports = this.imports.getImports(currentPackage);
        if (!imports.isEmpty()) {
            for (String importLine : imports) {
output.write("import "+importLine+";\n");
output.write("");
        }
output.write("\n");
output.write("");
        }
    }

    protected void generateCopyright(Writer output) throws IOException {
        String copyright = EugengoUtils.getCopyright(model);
        if (GeneratorUtil.notEmpty(copyright)) {
output.write(""+copyright+"\n");
output.write("");
        }
    }

    protected void generateClazzDocumentation(Writer output, ObjectModelClassifier classifier, String ... defaultDoc)
            throws IOException {
        generateDocumentation(output, classifier, "", defaultDoc);
    }

    protected void generateDocumentation(Writer output, ObjectModelElement element, String prefix, String ... defaultDoc)
            throws IOException {
        String doc = null;
        if (GeneratorUtil.hasDocumentation(element)) {
            doc = element.getDocumentation();
        } else {
            //TODO Manage defaultDoc
        }

        if (doc != null) {
            // Manage RC in the doc
            Pattern p = Pattern.compile("(\n)");
            Matcher m = p.matcher(doc);
            String docOk = m.replaceAll("\n"+prefix+" * ");
            
output.write(""+prefix+"/**\n");
output.write(""+prefix+" * "+docOk+"\n");
output.write(""+prefix+" */\n");
output.write("");
        }
    }

    /**
     * Generates a header for the given operation.
     * @param output The stream to write inside
     * @param op the operation which header is to generate
     * @param hasBody need to generate a body ?
     * <li>true (for classes) : generates ' {' at the end</li>
     * <li>false (for interfaces) : generates ';' at the end</li>
     * @param hasBody 
     * @throws IOException
     */
    protected void generateOperationHeader(Writer output, ObjectModelOperation op,
            boolean generateForInterface, String ... additionalExceptions) throws IOException {
        String opVisibility = op.getVisibility();
        //If generate for interface, only public methods are allowed
        if (generateForInterface && !"".equals(opVisibility) && !"public".equals(opVisibility)) {
            return;
        }
        String opName = op.getName();
output.write("    // Operation \""+opName+"\"\n");
output.write("");
        generateDocumentation(output, op, "    ");
        if (generateForInterface || "package".equals(opVisibility)) {
            opVisibility = "";
        } else {
            opVisibility += " ";
        }
        String opType = computeType(op.getReturnParameter());
        opType = getType(opType);
        String opAbstract = "";
        if (!generateForInterface && op.isAbstract()) {
            opAbstract = "abstract ";
        }
output.write("    "+opVisibility+""+opAbstract+""+opType+" "+opName+"(");
        boolean isFirst = true;
        for (ObjectModelParameter opParam : op.getParameters()) {
            String paramName = opParam.getName();
            String paramType = computeType(opParam);
            paramType = getType(paramType);
            if (!isFirst) {
output.write(", ");
            }
            isFirst = false;
output.write(""+paramType+" "+paramName+"");
        }
output.write(")");
        if ((op.getExceptions() != null && !op.getExceptions().isEmpty()) || (additionalExceptions != null && additionalExceptions.length > 0)) {
output.write(" throws");
            isFirst = true;
            Set<String> exceptions = new LinkedHashSet<String>();

            if (additionalExceptions != null) {
                for (String exception : additionalExceptions) {
                    exceptions.add(exception);
                }
            }
            if (op.getExceptions() != null) {
                for (String exception : op.getExceptions()) {
                    exceptions.add(exception);
                }
            }
            for (String exception : exceptions) {
                exception = getType(exception);
                if (!isFirst) {
output.write(",");
                }
                isFirst = false;
output.write(" "+exception+"");
            }
        }
        if (generateForInterface || op.isAbstract()) {
output.write(";\n");
output.write("\n");
output.write("");
        } else {
output.write(" {\n");
output.write("");
        }
    }

    /**
     * Generates a ioc name and injection. Will generate the class attribute
     * and getter/setter.
     * The name used is the name specified in the dependency class name.
     * @param output The stream to write inside
     * @param dep the dependency to generate.
     * @throws IOException
     */
    protected void generateIocDependency(Writer output, ObjectModelDependency dep)
            throws IOException {
        ObjectModelClassifier supplier = dep.getSupplier();
        if (supplier == null || EugengoUtils.isDao(supplier)) {
            return;
        }
        String supplierType = getType(supplier.getQualifiedName());
        String supplierVarName = EugengoUtils.toLowerCaseFirstLetter(supplier.getName());
        String supplierMethodSuffix = EugengoUtils.toUpperCaseFirstLetter(supplier.getName());
output.write("    // Dependency injection for \""+supplierVarName+"\"\n");
output.write("    private "+supplierType+" "+supplierVarName+";\n");
output.write("\n");
output.write("    public "+supplierType+" get"+supplierMethodSuffix+"() {\n");
output.write("        return "+supplierVarName+";\n");
output.write("    }\n");
output.write("\n");
output.write("    public void set"+supplierMethodSuffix+"("+supplierType+" "+supplierVarName+") {\n");
output.write("        this."+supplierVarName+" = "+supplierVarName+";\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

    protected void generateAttributesDeclaration(Writer output,
            ObjectModelClass clazz) throws IOException {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isNavigable() && !attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                generateAttributeDeclaration(output, attr);
            }
        }
    }

    protected void generateAttributeDeclaration(Writer output, ObjectModelAttribute attr)
            throws IOException {
        String attrType = computeType(attr);
        if (EugengoUtils.notEmpty(attrType)) {
            attrType = getType(attrType);
        } else {
            return;
        }
        String attrVisibility = attr.getVisibility();
        String attrName = attr.getName();
output.write("    // Declaration of attribute \""+attrName+"\"\n");
output.write("");
        generateDocumentation(output, attr, "    ");
        String value = computeDefaultValue(attr);
output.write("    "+attrVisibility+" "+attrType+" "+attrName+""+value+";\n");
output.write("\n");
output.write("");
    }

    /**
     * Compute correct type of param. If param is
     * <li> null : void
     * <li> cardinality > 1 : Collection of type
     * <li> cardinality > 1 and ordered: List of type
     * <li> cardinality > 1 and unique: Set of type
     * <li> other : the type
     * @param param
     * @return
     */
    protected String computeType(ObjectModelParameter param) {
        if (param == null) {
            return "void";
        }
        String result = param.getType();

        // if type is Wikitty then we used String
        if ("org.sharengo.wikitty.Wikitty".equals(result)
                || "Wikitty".equals(result)) {
            result = "String";
        }

        boolean isCollection = (param.getMaxMultiplicity() != 0
                && param.getMaxMultiplicity() != 1);
        if (isCollection) {
            Class<?> type = Collection.class;
            if (param.isOrdered()) {
                type = List.class;
            }
            if (param.isUnique()) {
                type = Set.class;
            }
            result = type.getName() + "<" + result + ">";
        }
        return result;
    }

    protected void generateAttributesAccessors(Writer output,
            ObjectModelClass clazz) throws IOException {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isNavigable() && !attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                generateAttributeAccessors(output, attr);
            }
        }
    }

    protected void generateAttributeAccessors(Writer output, ObjectModelAttribute attr)
            throws IOException {
        String attrType = computeType(attr);
        if (EugengoUtils.notEmpty(attrType)) {
            attrType = getType(attrType);
        } else {
            return;
        }
        String attrName = attr.getName();
        String attrNameCapitalized = EugengoUtils.toUpperCaseFirstLetter(attrName);
output.write("    // Accessors for attribute \""+attrName+"\"\n");
output.write("    public void set"+attrNameCapitalized+"("+attrType+" "+attrName+") {\n");
output.write("        this."+attrName+" = "+attrName+";\n");
output.write("    }\n");
output.write("\n");
output.write("    public "+attrType+" get"+attrNameCapitalized+"() {\n");
output.write("        return this."+attrName+";\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

    protected void generateStaticAttributes(Writer output, ObjectModelClass clazz)
            throws IOException {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isStatic() && "public".equals(attr.getVisibility())) {
                String type = computeType(attr);
                type = getType(type);
                String name = attr.getName();
                String value = computeDefaultValue(attr);
output.write("    // static attribute \""+name+"\"\n");
output.write("    public static "+type+" "+name+""+value+";\n");
output.write("\n");
output.write("");
            }
        }
    }

    protected String computeDefaultValue(ObjectModelAttribute attr) {
        String result = "";
        String value = attr.getDefaultValue();
        if (value != null) {
            String type = computeType(attr);
            type = getType(type);
            if ("String".equals(type)) {
                result = "\"" + value + "\"";
            } else if ("boolean".equalsIgnoreCase(type)) {
                result = "Boolean." + ("true".equalsIgnoreCase(value) + "").toUpperCase();
            } else if ("byte".equalsIgnoreCase(type) || "short".equalsIgnoreCase(type) || "int".equalsIgnoreCase(type) || "integer".equalsIgnoreCase(type)) {
                result = value;
            } else if ("long".equalsIgnoreCase(type)) {
                result = value + "L";
            } else if ("float".equalsIgnoreCase(type)) {
                result = value + "F";
            } else if ("double".equalsIgnoreCase(type)) {
                result = value + "D";
            } else if ("Date".equals(type)) {
                try {
                    Date d = new SimpleDateFormat().parse(value);
                    result = "new Date(" + d.getTime() + "l)";
                } catch (ParseException pe) {
                    log.warn("Unable to parse date", pe);
                    // Nothing else to do
                }
            } else {
                result = value;
            }
            result = " = " + result;
        }
        return result;
    }

    protected void generateDefaultConstructor(Writer output, String name)
            throws IOException {
output.write("    /**\n");
output.write("     * Default constructor \n");
output.write("     */\n");
output.write("    public "+name+"() {\n");
output.write("        super();\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

    protected void generateExceptionConstructors(Writer output, ObjectModelClass clazz)
            throws IOException {
        String name = clazz.getName();
        generateDefaultConstructor(output, name);

output.write("    public "+name+"(Throwable cause) {\n");
output.write("        super();\n");
output.write("        initCause(cause);\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

    protected void generateFullConstructor(Writer output, 
            String name, Collection<ObjectModelAttribute> attrs) throws IOException {
        int nb = 0;
        int total = 0;
        if (hasNavigableAndNonStaticAttributes(attrs)) {
output.write("    /**\n");
output.write("     * Constructor with all parameters initialized\n");
output.write("     * \n");
output.write("");
            for (ObjectModelAttribute attr : attrs) {
                if (attr.isNavigable() && !attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                    total ++;
                    String attName = attr.getName();
output.write("     * @param "+attName+"\n");
output.write("");                
                    }
                }
            
output.write("     */\n");
output.write("    public "+name+"(");
            for (ObjectModelAttribute attr : attrs) {
                if (attr.isNavigable() && !attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                    String attrName = attr.getName();
                    String attrType = getType(computeType(attr));
                    if (EugengoUtils.notEmpty(attrType)) {
                        attrType = getType(attrType);
                    } else {
                        return;
                    }
output.write(""+attrType+" "+attrName+"");
                    nb ++;
                    if (nb < total){
output.write(", ");                    
                    }
                }
            }
output.write(") {\n");
output.write("        super();\n");
output.write("");
            for (ObjectModelAttribute attr : attrs) {
                if (attr.isNavigable() && !attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                    String attrName = attr.getName();
output.write("        this."+attrName+" = "+attrName+";\n");
output.write("");
                }
            }

output.write("    } \n");
output.write("\n");
output.write("");
        }
    }

    protected boolean hasNavigableAndNonStaticAttributes(ObjectModelClass clazz) {
        return hasNavigableAndNonStaticAttributes(clazz.getAttributes());
    }

    protected boolean hasNavigableAndNonStaticAttributes(Collection<ObjectModelAttribute> attrs) {
        if (attrs != null && !attrs.isEmpty()) {
            for (ObjectModelAttribute attr : attrs) {
                if (attr.isNavigable() && !attr.isStatic()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Run throw the given ObjectModelClass and declare as an import each found
     * attribute type
     * @param clazz the class to run throw
     */
    protected void lookForAttributeImports(ObjectModelClass clazz) {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isNavigable() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                String type = computeType(attr);
                addImport(type);
            }
        }
    }

    /**
     * Run throw the given ObjectModelClass and declare as an import each found
     * static attribute type
     * @param clazz the class to run throw
     */
    protected void lookForStaticAttributeImports(ObjectModelClass clazz) {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isNavigable() && attr.isStatic() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                String type = computeType(attr);
                addImport(type);
            }
        }
    }

    /**
     * Run throw the given ObjectModelClassifier and declare each type found on
     * the operation declaration (return type, parameters type, exception
     * thrown)
     * @param classifier the classifier to run throw
     */
    protected void lookForOperationImports(ObjectModelClassifier classifier) {
        for (ObjectModelOperation op : classifier.getOperations()) {
            String returnType = computeType(op.getReturnParameter());
            addImport(returnType);
            for (ObjectModelParameter param : op.getParameters()) {
                String paramType = computeType(param);
                addImport(paramType);
            }
            for (String exceptionType : op.getExceptions()) {
                addImport(exceptionType);
            }
        }
    }

    /**
     * Run throw the given ObjectModelClassifier and declare as an import each
     * dependency's type found. The import is added only if the dependency's
     * supplier is a service or a dao
     * @param classifier the classifier to run throw
     */
    protected void lookForIocImports(ObjectModelClassifier classifier) {
        for (ObjectModelDependency dep : classifier.getDependencies()) {
            ObjectModelClassifier supplier = dep.getSupplier();
            if (supplier != null && EugengoUtils.isService(supplier)) {
                addImport(supplier.getQualifiedName());
            }
        }
    }

    /**
     * Look on the model for a tag value that indicates an implementation for
     * a specific datatype
     * @param type the type to look for a declared implementation
     * @return the found type or the original type
     */
    protected String checkForDatatype(String type) {
        if (type != null) {
            // Look for simple dataType
            String tag = model.getTagValue(EugengoConstants.PREFIX_DATATYPE + type);
            if (tag != null) {
                return tag;
            }
            // Look for generic dataType
            int idx = type.indexOf("<");
            if (idx != -1) {
                tag = model.getTagValue(EugengoConstants.PREFIX_DATATYPE + type.substring(0, idx));
            }
            if (tag != null) {
                return tag;
            }
        }
        // No dataType found, return type
        return type;
    }

    /**
     * Run throw the superclasses to get the first one.
     * 
     * @param clazz the class to run throw
     * @return the first found superClass or null
     */
    protected ObjectModelClass findSuperClass(ObjectModelClass clazz) {
        if (clazz.getSuperclasses() != null && !clazz.getSuperclasses().isEmpty()) {
            return clazz.getSuperclasses().iterator().next();
        }
        return null;
    }

    protected Collection<ObjectModelClass> findSubClasses(ObjectModelClass clazz) {
        Collection<ObjectModelClass> result = new ArrayList<ObjectModelClass>();
        for (ObjectModelClass potentialSubClass : model.getClasses()) {
            if (clazz.equals(findSuperClass(potentialSubClass))) {
                result.add(potentialSubClass);
            }
        }
        return result;
    }

    protected void generateHashCode(Writer output, ObjectModelClass clazz) throws IOException {
output.write("    public int hashCode() {\n");
output.write("        int result = 0;\n");
output.write("");
        String prefix = "";
        if (EugengoUtils.isEntity(clazz)) {
output.write("        if (id != null) {\n");
output.write("            result = id.hashCode();\n");
output.write("        } else {\n");
output.write("");
            prefix = "    ";
        }
        generateHashCodeFromAttributes(output, clazz, prefix);
        if (EugengoUtils.isEntity(clazz)) {
output.write("        }\n");
output.write("");
        }
output.write("        return result;\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

    private void generateHashCodeFromAttributes(Writer output, ObjectModelClass clazz, String prefix) throws IOException {
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (attr.isNavigable() && (attr.getStereotypes() == null || attr.getStereotypes().isEmpty())) {
                String attrName = attr.getName();
                String attrType = getType(attr.getType());
                if (EugengoUtils.isPrimitiveType(attr)) {
                    // If the leading character is an uppercased letter...
                    if (attrType.charAt(0) == attrType.toUpperCase().charAt(0)) {
output.write(""+prefix+"        if ("+attrName+" != null) {\n");
output.write(""+prefix+"            result = 29 * result + "+attrName+".hashCode();\n");
output.write(""+prefix+"        }\n");
output.write("");
                    } else {
                        attrType = EugengoUtils.toUpperCaseFirstLetter(attrType);
                        if ("Int".equals(attrType)) {
                            attrType = "Integer";
                        }
output.write(""+prefix+"        result = 29 * result + new "+attrType+"("+attrName+").hashCode();\n");
output.write("");
                    }
                } else {
output.write(""+prefix+"        if ("+attrName+" != null) {\n");
output.write(""+prefix+"            result = 29 * result + "+attrName+".hashCode();\n");
output.write(""+prefix+"        }\n");
output.write("");
                }
            }
        }
    }
}
