/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: ManualMigrationCallback.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.3/topia-service-migration/src/main/java/org/nuiton/topia/migration/ManualMigrationCallback.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.migration;

import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.jdbc.Work;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.util.StringUtil;
import org.nuiton.util.Version;
import static org.nuiton.i18n.I18n._;

/**
 * ManualMigrationCallback
 *
 * @author tchemit <chemit@codelutin.com>
 * @version $Revision: 1894 $
 *
 * Last update : $Date: 2010-04-15 17:44:51 +0200 (jeu., 15 avril 2010) $
 * By : */
public abstract class ManualMigrationCallback {

    /** Logger */
    static private Log log = LogFactory.getLog(ManualMigrationCallback.class);

    /**
     * CallbackHandler return type.
     */
    public enum MigrationChoice {
        // no migration

        NO_MIGRATION,
        // migration
        MIGRATION,
        // migration done by application, only version managed
        CUSTOM_MIGRATION
    }

    public abstract boolean askUser(Version dbVersion,
                                    Version applicationVersion,
                                    List<Version> versions);

    /**
     * Tentative de migration depuis la version de la base version la version
     * souhaitee.
     *
     * On applique toutes les migrations de version indiquee dans le parametre
     * <code>version</code>.
     *
     * Pour chaque version, on cherche la methode migrateTo_XXX ou XXX est la
     * version transforme en identifiant java via la methode
     * {@link Version#getValidName()} et on l'execute.
     *
     * Note: pour chaque version a appliquer, on ouvre une nouvelle transaction.
     *
     * @param ctxt topia context de la transaction en cours
     * @param dbVersion database version
     * @param applicationVersion application version
     * @param showSql drapeau pour afficher les requete sql
     * @param showProgression drapeau pour afficher la progression
     * @param versions  all versions knwon by service  @return migration a
     *                  ggrement
     * @return le choix de migration executee
     */
    public MigrationChoice doMigration(TopiaContext ctxt, Version dbVersion,
                                       Version applicationVersion,
                                       boolean showSql,
                                       boolean showProgression,
                                       List<Version> versions) {

        MigrationChoice result = MigrationChoice.NO_MIGRATION;

        boolean doMigrate = askUser(dbVersion, applicationVersion, versions);

        if (doMigrate) {
            TopiaContextImplementor tx;

            for (Version v : versions) {
                // ouverture d'une connexion direct JDBC sur la base
                try {

                    tx = (TopiaContextImplementor) ctxt.beginTransaction();

                    try {

                        String methodName = "migrateTo_" + v.getValidName();

                        Method m;
                        try {
                            m = getClass().getMethod(methodName,
                                    TopiaContextImplementor.class,
                                    boolean.class,
                                    boolean.class);

                            m.setAccessible(true);

                            log.info(_("topia.migration.start.migrate", v));

                            if (log.isDebugEnabled()) {
                                log.debug("launch method " + methodName);
                            }

                            m.invoke(this, tx, showSql, showProgression);


                        } catch (NoSuchMethodException e) {
                            // try with no boolean values (for legacy callbacks)
                            m = getClass().getMethod(methodName,
                                    TopiaContextImplementor.class);

                            m.setAccessible(true);

                            log.info(_("topia.migration.start.migrate", v));

                            if (log.isDebugEnabled()) {
                                log.debug("launch method " + methodName);
                            }

                            m.invoke(this, tx);

                        }

                        // commit des modifs
                        tx.commitTransaction();

                        // la migration a reussi, on dit que c fait
                        result = MigrationChoice.CUSTOM_MIGRATION;
                    } catch (Exception eee) {
                        // en cas d'erreur
                        log.error("Migration impossible de la base", eee);
                        // rollback du travail en cours
                        tx.rollbackTransaction();
                    } finally {
                        // close database connexion
                        if (tx != null) {
                            tx.closeContext();
                        }
                    }

                } catch (Exception eee) {
                    log.error("Error lors de la tentative de migration", eee);
                }
            }
        }
        return result;
    }

    public void executeSQL(TopiaContextImplementor tx, String... sqls)
            throws TopiaException {
        executeSQL(tx, false, false, sqls);
    }

    /**
     * Executes the given {@code sqls} requests.
     *
     * @param tx the session
     * @param showSql flag to see sql requests
     * @param showProgression flag to see progession on console
     * @param sqls requests to execute
     * @throws TopiaException if any pb
     * @since 2.3.0
     */
    public void executeSQL(TopiaContextImplementor tx,
                           final boolean showSql,
                           final boolean showProgression,
                           final String... sqls) throws TopiaException {

        if (log.isInfoEnabled()) {

            log.info(_("topia.migration.start.sqls", sqls.length));
        }
        if (showSql) {
            StringBuilder buffer = new StringBuilder();
            for (String s : sqls) {
                buffer.append(s).append("\n");
            }
                log.info("SQL TO EXECUTE :\n" +
                        "--------------------------------------------------------------------------------\n" +
                        "--------------------------------------------------------------------------------\n" +
                        buffer.toString() +
                        "--------------------------------------------------------------------------------\n" +
                        "--------------------------------------------------------------------------------\n"
                );
        }
        tx.getHibernate().doWork(new Work() {

            @Override
            public void execute(Connection connection) throws SQLException {
                int index = 0;
                int max = sqls.length;
                for (String sql : sqls) {
                    long t0 = System.nanoTime();
                    if (log.isInfoEnabled()) {
                        String message = "";

                        if (showProgression) {
                            message = _("topia.migration.start.sql", ++index, max);
                        }
                        if (showSql) {
                            message += "\n" + sql;
                        }
                        if (showProgression || showSql) {

                            log.info(message);
                        }
                    }
                    PreparedStatement sta = connection.prepareStatement(sql);
                    sta.executeUpdate();
                    sta.close();
                    if (log.isDebugEnabled()) {
                        String message;
                        message = _("topia.migration.end.sql", ++index, max, StringUtil.convertTime(System.nanoTime() - t0));
                        log.debug(message);
                    }
                }
            }
        });

    }
}
