/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.finalizers;

import jaxx.compiler.java.JavaArgument;
import jaxx.compiler.java.JavaField;
import jaxx.compiler.java.JavaFile;
import jaxx.compiler.java.JavaFileGenerator;
import jaxx.compiler.*;
import jaxx.compiler.CompiledObject.ChildRef;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorLoader;
import jaxx.compiler.tags.validator.BeanValidatorHandler;
import jaxx.compiler.tags.validator.BeanValidatorHandler.CompiledBeanValidator;

import java.util.List;
import jaxx.runtime.SwingUtil;

/** @author chemit */
public class ValidatorFinalizer implements JAXXCompilerFinalizer {

    protected static final JavaField VALIDATOR_IDS_FIELD = JavaFileGenerator.newField(java.lang.reflect.Modifier.PROTECTED,
            "java.util.List<String>", "validatorIds", true, "new ArrayList<String>()");

    @Override
    public void finalizeCompiler(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) {

        if (!BeanValidatorHandler.hasValidator(compiler)) {
            return;
        }

        for (CompiledObject object : compiler.getObjects().values()) {
            List<ChildRef> childs = object.getChilds();
            if (childs == null || childs.isEmpty()) {
                continue;
            }
            for (ChildRef child : childs) {
                String javaCode = child.getChildJavaCode();
                // some validators are defined on this object
                boolean found = BeanValidatorHandler.isComponentUsedByValidator(compiler, child.getChild().getId());
                if (found) {
                    // box the child component in a JxLayer
                    child.setChildJavaCode(SwingUtil.class.getName() + ".boxComponentWithJxLayer(" + javaCode + ")");
                }
            }
        }
        String eol = JAXXCompiler.getLineSeparator();
        // register validator
        compiler.appendLateInitializer("// validator setup" + eol);
        for (CompiledBeanValidator validator : BeanValidatorHandler.getValidators(compiler)) {
            String id = compiler.getJavaCode(validator.getId());
            compiler.appendLateInitializer("validatorIds.add(" + id + ");");
            compiler.appendLateInitializer(eol);
            compiler.appendLateInitializer("getValidator(" + id + ").installUIs();");
            compiler.appendLateInitializer(eol);
            compiler.appendLateInitializer("getValidator(" + id + ").reloadBean();");
            //compiler.appendLateInitializer("getValidator(" + id + ").validate();");
            compiler.appendLateInitializer(eol);
        }
        compiler.appendLateInitializer("validatorIds = java.util.Collections.unmodifiableList(validatorIds);");
        compiler.appendLateInitializer(eol);
    }

    @Override
    public void prepareJavaFile(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) throws ClassNotFoundException {
        if (!BeanValidatorHandler.hasValidator(compiler)) {
            return;
        }
        Class<?> validatorClass = compiler.getConfiguration().getValidatorClass();
        String validatorFQN = validatorClass.getName();
        javaFile.addImport(validatorFQN);

        //TODO use the specific JAXXValidator interface (swing, gwt,...)
        Class<?> validatorInterface = jaxx.runtime.JAXXValidator.class;

        if (javaFile.isSuperclassIsJAXXObject()) {
            ClassDescriptor superClass = ClassDescriptorLoader.getClassDescriptor(javaFile.getSuperClass());
            boolean parentIsValidator = ClassDescriptorLoader.getClassDescriptor(validatorInterface).isAssignableFrom(superClass);

            if (parentIsValidator) {
                // nothing to generate (use the parent directly)
                return;
            }
        }

        // add JAXXValidator interface
        javaFile.addInterface(JAXXCompiler.getCanonicalName(validatorInterface));

        // implements JAXXValidator
        javaFile.addField(VALIDATOR_IDS_FIELD);
        javaFile.addMethod(JavaFileGenerator.newMethod(
                java.lang.reflect.Modifier.PUBLIC,
                validatorFQN + "<?>",
                "getValidator",
                "return (" + validatorFQN + "<?>) (validatorIds.contains(validatorId) ? getObjectById(validatorId) : null);",
                true,
                new JavaArgument("String", "validatorId")));
    }
}
