/*
 * *##% 
 * I18n :: Maven Plugin
 * Copyright (C) 2007 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%* */
package org.nuiton.i18n.plugin.parser.impl;

import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.MirroredFileUpdater;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.xpath.XPath;
import java.io.File;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;

/**
 * Récupération des chaine à traduire depuis les fichiers xml de validation.
 * <p/>
 * Le goal doit etre execute avant que les resources soient copiees dans
 * target/classes pour rendre operatne le file updater (sinon lesfichiers sont
 * toujours a jour...)
 *
 * @author chemit
 * @goal parserValidation
 * @phase generate-resources
 */
public class ParserValidationMojo extends ParserXmlMojo {

    final URL xworksResource = getClass().getResource("/xwork-validator-1.0.2.dtd");

    /**
     * Source entries (src+includes+excludes) .
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/**-validation.xml"
     */
    protected String defaultIncludes;

    /**
     * Where jaxx files should have been generated.
     *
     * @parameter expression="${i18n.cp}" default-value="${basedir}/target/classes"
     */
    protected File cp;

    /**
     * Regles xml.
     *
     * @parameter expression="${i18n.rulesValidation}" default-value="validation.rules"
     */
    protected String rulesValidation;

    /**
     * default src for an entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/resources"
     * @required
     */
    protected File defaultBasedir;

    /**
     * Always use the local xowrks dtd to increase performance.
     *
     * @parameter expression="${i18n.useLocalResolver}" default-value="true"
     * @since 1.6.0
     */
    protected boolean useLocalResolver;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return new MirroredFileUpdater("", "", entry.getBasedir(), cp) {

            @Override
            public File getMirrorFile(File f) {
                String file =
                        f.getAbsolutePath().substring(prefixSourceDirecotory);
                return new File(destinationDirectory + File.separator + file);
            }
        };
    }

    @Override
    protected String getOutGetter() {
        return "validation.getter";
    }

    @Override
    protected String getFileRules() {
        return rulesValidation;
    }

    @Override
    protected String getCoreFileRules() {
        return "validation.rules";
    }

    protected XmlFileParser newXmlFileParser(final XPath xpath,
                                             final DocumentBuilder builder) {

        // add cached entity resolver
        builder.setEntityResolver(getEntityResolver());

        return new XmlFileParser(getLog(),
                                 encoding,
                                 oldParser,
                                 showTouchedFiles,
                                 rules,
                                 xpath,
                                 builder) {
            @Override
            public String extract(String i18nString) {
                String s = null;
                if (!i18nString.trim().isEmpty()) {
                    s = i18nString.trim();
                    int end = s.indexOf("##");
                    if (end > 0) {
                        // remove params from key
                        s = s.substring(0, end);
                    }
                }
                if (getLog().isDebugEnabled()) {
                    getLog().debug(i18nString + " = " + s);
                }
                return s;
            }
        };
    }

    protected EntityResolver getEntityResolver() {

        return new EntityResolver() {

            public static final String XWORK_PUBLIC_ID =
                    "-//OpenSymphony Group//XWork Validator 1.0.2//EN";

            boolean useLocal = useLocalResolver;

            @Override
            public InputSource resolveEntity(String publicId, String systemId)
                    throws SAXException, IOException {
                if (getLog().isDebugEnabled()) {
                    getLog().debug("publicID:" + publicId + ", systemId:" +
                                   systemId);
                }
                if (XWORK_PUBLIC_ID.equals(publicId)) {
                    if (!useLocal) {
                        URL uri = new URL(systemId);
                        if (verbose) {
                            getLog().info("try to connect to " + uri);
                        }
                        URLConnection openConnection = uri.openConnection();
                        openConnection.setUseCaches(true);
                        openConnection.setConnectTimeout(1000);
                        try {
                            openConnection.connect();
                            return new InputSource(
                                    openConnection.getInputStream());
                        } catch (SocketTimeoutException e) {
                            useLocal = true;
                        } catch (IOException e) {
                            useLocal = true;
                        }
                    }

                    // use directly local resource
                    InputSource inputSource =
                            new InputSource(xworksResource.openStream());
                    return inputSource;
                }
                // use the default behaviour
                return null;
            }
        };
    }

    @Override
    @Deprecated
    protected String getKeyModifierStart() {
        return "=\\s*[\"\']";
    }

    @Override
    @Deprecated
    protected String getKeyModifierEnd() {
        return "[\"\']";
    }
}
