/*
 * *##% 
 * I18n :: Maven Plugin
 * Copyright (C) 2007 - 2010 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.i18n.plugin;

import org.nuiton.io.SortedProperties;

import java.io.File;
import java.io.FileOutputStream;
import java.util.Arrays;
import java.util.Locale;

/**
 * Merge des fichiers de propriétés avec les anciens.
 *
 * @author julien
 * @goal gen
 * @phase generate-resources
 * @execute goal=get
 */
public class GenerateMojo extends AbstractI18nMojo {

    /**
     * Un drapeau pour vérifier que les bundles ne contiennent pas d'entrées
     * vides.
     *
     * @parameter expression="${i18n.checkBundle}" default-value="true"
     * @required
     * @since 1.0.0
     */
    protected boolean checkBundle;

    /**
     * Un drapeau pour afficher les entrées vides. (nécessite {@link
     * #checkBundle} activé).
     *
     * @parameter expression="${i18n.showEmpty}" default-value="false"
     * @required
     * @since 1.0.0
     */
    protected boolean showEmpty;

    /**
     * conserve les anciens fichiers de traduction avec un suffix ~
     * <p/>
     * Note: par defaut, on ne l'active pas.
     *
     * @parameter expression="${i18n.keepBackup}" default-value="false"
     */
    protected boolean keepBackup;

    /**
     * Conserve les fichiers de getters.
     *
     * @parameter expression="${i18n.keepGetters}" default-value="false"
     */
    protected boolean keepGetters;


    @Override
    protected boolean checkSkip() {
        if (!needGeneration()) {
            getLog().info("No getter detected - all files are up to date.");
            return false;
        }
        return true;
    }

    @Override
    protected void doAction() throws Exception {
        if (!silent) {
            getLog().info("config - src basedir : " + src.getAbsolutePath());
            getLog().info("config - out basedir : " + out.getAbsolutePath());
            getLog().info("config - locales     : " + Arrays.toString(locales));
        }
        for (Locale locale : locales) {
            if (!silent) {
                getLog().info("prepare bundle for locale " + locale);
            }
            // Merge
            File bundleSrc = getI18nFile(src, artifactId, locale, false);
            File bundleOut = getI18nFile(out, artifactId, locale, false);
            File bundleGetterOut = getI18nFile(
                    out, artifactId + GetterMojo.FROM_GETTERS, locale, false);

            SortedProperties propertiesSrc = new SortedProperties(encoding);

            if (bundleSrc.exists()) {
                propertiesSrc.load(bundleSrc);
            }

            SortedProperties propertiesOut = new SortedProperties(encoding);

            if (!strictMode) {
                // si on n'est pas en mode strict, on doit push back in
                //  bundle out, all the bundle src keys
                propertiesOut.putAll(propertiesSrc);
            }
            propertiesOut.load(bundleGetterOut);

            // Parcours des clés
            for (Object key : propertiesOut.keySet()) {
                Object oldKey = propertiesOut.get(key);
                Object value = propertiesSrc.get(oldKey);

                // Récupération de la clé si elle a été renommée
                if (!key.equals(oldKey) && value == null) {
                    value = propertiesSrc.get(key);
                }

                if (value != null) {
                    propertiesOut.put(key, value);
                } else {
                    propertiesOut.put(key, "");
                }
            }

            //fixme : on devrait laisser le fichier en utf8 ?
            FileOutputStream outStream = new FileOutputStream(bundleOut);
            try {
                propertiesOut.store(outStream);
            } finally {
                outStream.close();
            }

            // Sauvegarde avant copie
            if (genSrc && keepBackup) {
                backupFile(bundleSrc);
            }
            if (!silent) {
                getLog().info("merge bundle " + locale + " to out");
            }

            if (checkBundle) {
                checkBundle(locale, propertiesOut, showEmpty);
            }

            if (genSrc) {
                // Copie des fichiers dans les sources
                copyFile(bundleOut, bundleSrc);
                if (!silent) {
                    getLog().info("copy bundle " + locale + " to src");
                }
            }

            if (!keepGetters) {
                deleteFile(bundleGetterOut);
            }
        }
    }
}
