/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.lang.reflect.Method;

import javassist.CtClass;
import javassist.CtMethod;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.Bytecode;
import javassist.bytecode.CodeAttribute;
import javassist.bytecode.ConstPool;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.Opcode;
import javassist.bytecode.StackMapTable;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.javassist.JavassistMethod;

/**
 * <p>Class to create implementations of the {@link JavassistMethod} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class JavassistMethodFactory extends JavassistBehaviorFactory
{
   /** The names of the interfaces we are implementing */ 
   private static final String[] interfaceNames;
   
   /** The methods from the interface that are being implemented */ 
   private static final Method[] methods;
   
   static
   {
      interfaceNames = new String[] {JavassistMethod.class.getName()};
      methods = new Method[1];
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(JavassistMethod.class, "invoke", Object.class, Object[].class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      } 
   }

   /** The method we are targeting */
   private final CtMethod ctMethod;
   
   /** The name of the class being generated */
   private final String className = JavassistMethod.class.getName() + counter.incrementAndGet();
   
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param ctMethod the constructor we are generating a {@link JavassistMethod} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistMethodFactory(Class<?> superClass, CtMethod ctMethod, boolean debug)
   {
      super(superClass, ctMethod, debug);
      this.ctMethod = ctMethod;
   }

   String getGeneratedClassName()
   {
      return className;
   }
   

   String[] getInterfaceNames()
   {
      return interfaceNames;
   }

   /**
    * The return type of the method we are targeting
    * 
    * @return the target type
    */
   private CtClass getReturnType()
   {
      try
      {
         return ctMethod.getReturnType();
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
   }

   @Override
   MethodInfo implementMethod(int index, ConstPool cp)
   {
      if (index >= methods.length)
         return null;

      Method method = methods[index];
      
      String desc = RuntimeSupport.makeDescriptor(method);
      MethodInfo minfo = new MethodInfo(cp, method.getName(), desc);
      minfo.setAccessFlags(Modifier.PUBLIC);
      setThrows(minfo, cp, method.getExceptionTypes());
      Bytecode code = new Bytecode(cp, 0, 0);
      
      int pc = code.currentPc();
      

      boolean isStatic = Modifier.isStatic(ctBehavior.getModifiers());

      if (!isStatic)
      {
         //push and cast the target object
         code.addAload(1);
         code.addCheckcast(ctMethod.getDeclaringClass());
      }
      
      addParameters(code, getParameterTypes(), 2);

      if (isStatic)
         code.addInvokestatic(ctMethod.getDeclaringClass(), ctMethod.getName(), ctMethod.getSignature());
      else if (ctMethod.getDeclaringClass().isInterface())
         code.addInvokeinterface(ctMethod.getDeclaringClass(), ctMethod.getName(), ctMethod.getSignature(), countParameterStackSize(0, getParameterTypes()));
      else
         code.addInvokevirtual(ctMethod.getDeclaringClass(), ctMethod.getName(), ctMethod.getSignature());
      
      if (getReturnType() == CtClass.voidType)
      {            
         //return null;
         code.addOpcode(Opcode.ACONST_NULL);
         code.addOpcode(Opcode.ARETURN);
      }
      else
      {
         boxReturnValue(code, getReturnType());
         code.addOpcode(Opcode.ARETURN);
      }

      //We need 3 local variable slots.
      //One for 'this', one for the target reference and one for the argument array.
      //These are all object references and so take one slot each 
      code.setMaxLocals(3);
      
      CodeAttribute ca = code.toCodeAttribute();
      minfo.setCodeAttribute(ca);

      StackMapTable.Writer writer = new StackMapTable.Writer(32);
      writer.sameFrame(pc);
      ca.setAttribute(writer.toStackMapTable(cp));
      return minfo;
   }
}