/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.util.Arrays;

import javassist.CtConstructor;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.spi.ConstructorInfo;
import org.jboss.reflect.spi.MutableConstructorInfo;
import org.jboss.util.JBossStringBuilder;

/**
 * ConstructorInfo that relies on Javassist to answer reflective queries and to invoke
 * the represented constructor.
 * 
 *   This class also relies on Javassist to perform the instrumentation operations defined in
 *  {@code MutableConstructorInfo}
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 104251 $
 * @see MutableConstructorInfo
 */
// TODO: review hierarchy structure. Shouldn't JavassistConstructorInfo extend ConstructorInfoImpl, just like
// ReflectConstructorInfoImpl?
public class JavassistConstructorInfo extends JavassistAnnotatedParameterInfo implements MutableConstructorInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -2255405601790592604L;

   /** The constructor implementation */
   private transient volatile JavassistConstructor constructor;

   /**
    * Create a new JavassistConstructorInfo.
    * 
    * @param annotationHelper the annotation helper
    * @param typeInfo the type info
    * @param ctConstructor the constructor
    */
   public JavassistConstructorInfo(AnnotationHelper annotationHelper, JavassistTypeInfo typeInfo, CtConstructor ctConstructor)
   {
      super(annotationHelper, typeInfo, ctConstructor);
   }

   public Object newInstance(Object[] args) throws Throwable
   {
      if (constructor == null)
      {
         JavassistConstructor ctor = JavassistReflectionFactory.INSTANCE.createConstructor(this);
         if (constructor == null)
            constructor = ctor; 
      }
      
      JavassistAccessController.checkAccess(this);
      
      return constructor.newInstance(args);
   }

   @Override
   protected int getHashCode()
   {
      int result = getDeclaringClass().hashCode();
      if (parameters == null)
         generateParameters();
      if (parameterTypes != null)
      {
         for (int i = 0; i < parameterTypes.length; i++)
            result = 29 * result + parameterTypes[i].hashCode();
      }
      return result;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (this == obj) 
         return true;
      if (obj == null || obj instanceof ConstructorInfo == false)
         return false;

      final ConstructorInfo other = (ConstructorInfo) obj;
      
      if (getDeclaringClass().equals(other.getDeclaringClass()) == false)
         return false;
      return (Arrays.equals(getParameterTypes(), other.getParameterTypes()));
   }


   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append(Arrays.asList(getParameterTypes()));
      super.toString(buffer);
   }
   
   protected CtConstructor getCtConstructor()
   {
      return (CtConstructor)ctBehavior;
   }
}
