package org.immutables.value.processor.encode;

import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * NamingMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.encode.Encoding.Naming}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.encode.Mirrors.Naming} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class NamingMirror implements Mirrors.Naming {
  public static final String QUALIFIED_NAME = "org.immutables.encode.Encoding.Naming";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.encode.Mirrors.Naming";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Naming";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code NamingMirror}, present if this annotation found
   */
  public static Optional<NamingMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code NamingMirror}, present if this annotation found
   */
  public static Optional<NamingMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new NamingMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code NamingMirror}s
   */
  public static ImmutableList<NamingMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<NamingMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new NamingMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code NamingMirror}
   */
  public static NamingMirror from(TypeElement element) {
    return new NamingMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code NamingMirror}, present if mirror matched this annotation type
   */
  public static Optional<NamingMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String value;
  private final boolean depluralize;
  private final StandardNaming standard;

  private NamingMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String value = null;
    boolean depluralize = false;
    StandardNaming standard = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Naming");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("depluralize".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Naming");
        }
        DepluralizeExtractor depluralizeExtractor$ = new DepluralizeExtractor();
        annotationValue$.accept(depluralizeExtractor$, null);

        depluralize = depluralizeExtractor$.get();
        continue;
      }
      if ("standard".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Naming");
        }
        StandardExtractor standardExtractor$ = new StandardExtractor();
        annotationValue$.accept(standardExtractor$, null);

        standard = standardExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "default attribute 'value'");
    this.depluralize = depluralize;
    this.standard = Preconditions.checkNotNull(standard, "default attribute 'standard'");
  }

  private NamingMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String value = null;
    boolean depluralize = false;
    StandardNaming standard = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @Naming");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        continue;
      }
      if ("depluralize".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'depluralize' attribute of @Naming");
        }
        DepluralizeExtractor depluralizeExtractor$ = new DepluralizeExtractor();
        annotationValue$.accept(depluralizeExtractor$, null);

        depluralize = depluralizeExtractor$.get();
        continue;
      }
      if ("standard".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'standard' attribute of @Naming");
        }
        StandardExtractor standardExtractor$ = new StandardExtractor();
        annotationValue$.accept(standardExtractor$, null);

        standard = standardExtractor$.get();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "value for 'value'");
    this.depluralize = depluralize;
    this.standard = Preconditions.checkNotNull(standard, "value for 'standard'");
  }

  /**
   * @return value of attribute {@code value}
   */
  @Override
  public String value() {
    return value;
  }

  /**
   * @return value of attribute {@code depluralize}
   */
  @Override
  public boolean depluralize() {
    return depluralize;
  }

  /**
   * @return value of attribute {@code standard}
   */
  @Override
  public StandardNaming standard() {
    return standard;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Naming.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return Mirrors.Naming.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ value.hashCode();
    h += 127 * "depluralize".hashCode() ^ Booleans.hashCode(depluralize);
    h += 127 * "standard".hashCode() ^ standard.hashCode();
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof NamingMirror) {
      NamingMirror otherMirror = (NamingMirror) other;
      return value.equals(otherMirror.value)
          && depluralize == otherMirror.depluralize
          && standard.equals(otherMirror.standard);
    }
    return false;
  }

  @Override
  public String toString() {
    return "NamingMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }

  private static class DepluralizeExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'depluralize' in @" + QUALIFIED_NAME);
    }
  }

  private static class StandardExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    StandardNaming value;

    @Override
    public Void visitEnumConstant(VariableElement value, Void p) {
      this.value = StandardNaming.valueOf(value.getSimpleName().toString());
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    StandardNaming get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'standard' in @" + QUALIFIED_NAME);
    }
  }
}
