/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.io.File;
import java.text.DateFormat;
import java.util.Date;
import java.util.Locale;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.PersistentLocale;
import org.apache.tapestry5.services.Request;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.services.ServiceAuth;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.services.Configuration;

/**
 * Classe du Layout Component Border.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: Border.java 2755 2009-08-26 10:01:22Z nrannou $
 */
@IncludeStylesheet("context:css/main.css")
public class Border {

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "borderFeedback")
    private FeedBack feedback;

    /**
     * Objet de session (SSO) représentant l'adresse du site.
     */
    @SuppressWarnings("unused")
    @SessionState
    private String siteURL;

    /**
     * Adresse du fil de syndication
     */
    @SuppressWarnings("unused")
    @Parameter(required = false, defaultPrefix = BindingConstants.LITERAL)
    @Property
    private String feedUrl;

    /**
     * Hierarchie de l'adresse
     */
    @SuppressWarnings("unused")
    @Parameter(required = false)
    @Property
    private Lien[] address;

    /**
     * Chaine définissant le style de la page
     */
    @Parameter(required = true, defaultPrefix = BindingConstants.LITERAL)
    @Property
    private String pageLogo;

    /**
     * Chaine definissant le logo de la page
     */
    @SuppressWarnings("unused")
    @Property
    private String logo;

    /**
     * Lien pour l'itération
     */
    @SuppressWarnings("unused")
    @Property
    private Lien lien;

    /**
     * Champ login du formulaire d'identification
     */
    @Property
    private String login;

    /**
     * Champ password du formulaire d'identification
     */
    @Property
    private String password;

    /**
     * Objet de session représentant l'utilisateur identifié
     */
    @SessionState
    @Property
    private UserDTO user;

    /**
     * vrai si l'utilisateur s'est identifié, faux sinon
     */
    @SuppressWarnings("unused")
    @Property
    private boolean userExists;

    /** Format des dates */
    @SuppressWarnings("unused")
    @Property
    private DateFormat dateFormat;

    /** Locale courante */
    @Inject
    private Locale currentLocale;

    /**
     * Locale permettant le changement de langue
     */
    @Inject
    private PersistentLocale localeService;

    /**
     * Accès à la requête
     */
    @Inject
    private Request request;

    /**
     * Accès aux messages
     */
    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /** Injection des services */
    @Inject
    private ServiceAuth serviceAuth;

    /**
     * Methode appelée lorsque l'utilisateur s'identifie. L'utilisateur est
     * retrouvé grâce au service d'authentification.
     *
     * @return la zone à mettre à jour
     */
    Object onSuccessFromLoginForm() {
        UserDTO userDTO = serviceAuth.getUser(login, MD5.encode(password));
        if (userDTO != null) {
            user = userDTO;
            feedback.addInfo(messages.get("loginOk"));
        } else {
            feedback.addError(messages.get("loginFailed"));
        }
        return null;
    }

    /**
     * Methode appelée lorsque l'utilisateur se déconnecte. L'objet user est
     * désinstancié (=null).
     *
     * @return la zone à mettre à jour
     */
    Object onActionFromLogout() {
        user = null;
        return "Index";
    }

    /**
     * Méthode appelée lorsque l'utilisateur choisit la langue française
     */
    void onActionFromLocaleFr() {
        localeService.set(Locale.FRENCH);
    }

    /**
     * Méthode appelée lorsque l'utilisateur choisit la langue anglaise
     */
    void onActionFromLocaleEn() {
        localeService.set(Locale.ENGLISH);
    }

    /**
     * Date et heure actuelles.
     */
    public Date getCurrentTime() {
        return new Date();
    }

    /**
     * Version de l'application.
     */
    public String getVersion() {
        return conf.getProperty("version");
    }

    /**
     * Retourne si l'utilisateur a renseigné un nom ou un prénom.
     */
    public boolean isUserNamed() {
        return (user.getFirstName() != null) || (user.getLastName() != null);
    }

    /**
     * Vérifie que le fichier de flux de syndication existe.
     *
     * @return vrai si le fichier existe
     */
    public boolean isFeedFileExisting() {
        if (feedUrl == null) {
            return false;
        }
        File feedFile = new File(feedUrl);
        return feedFile.exists() ? true : false;
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                DateFormat.SHORT, currentLocale);

        // URL du site
        siteURL = "http://" + request.getHeader("host")
                + request.getContextPath() + "/";

        // Logo
        if (Locale.FRENCH.equals(currentLocale)) {
            logo = "pollen" + pageLogo + "_fr.png";
        } else {
            logo = "pollen" + pageLogo + "_en.png";
        }
    }
}
