/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.util.ArrayList;
import java.util.List;

import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.ioc.annotations.Inject;

/**
 * Composant qui affiche des messages à l'utilisateur.
 *
 * Ce composant permet d'afficher des messages avec une apparence différente
 * selon leur type (info ou erreur). Lors de l'affichage de la page, les
 * messages seront affichés et la collection de messages sera vidée.
 *
 * Pour personnaliser l'apparence des messages il faut définir les classes
 * fb-error et fb-info dans le CSS.
 *
 * Utilisation :
 *
 * <t:feedback t:id="feedback"/>
 *
 * @Component(id = "feedback") private FeedBack feedback;
 *
 *               feedback.addInfo("message d'info");
 *               feedback.addError("message d'erreur");
 *
 * @author rannou
 * @version $Id: FeedBack.java 2615 2009-07-01 13:47:45Z nrannou $
 */
public class FeedBack {

    /** Messages d'erreur */
    private List<String> errorMessages;

    /** Messages d'info */
    private List<String> infoMessages;

    @Inject
    private ComponentResources _resources;

    public FeedBack() {
        errorMessages = new ArrayList<String>();
        infoMessages = new ArrayList<String>();
    }

    void beginRender(MarkupWriter writer) {

        // Rendu des messages d'erreur et vidage de la collection
        if (!errorMessages.isEmpty()) {
            writer.element("div", "class", "fb-error");
            for (String message : errorMessages) {
                writer.write(message);
                writer.element("br");
                writer.end();
            }
            writer.end();
        }
        errorMessages.clear();

        // Rendu des messages d'info et vidage de la collection
        if (!infoMessages.isEmpty()) {
            writer.element("div", "class", "fb-info");
            for (String message : infoMessages) {
                writer.write(message);
                writer.element("br");
                writer.end();
            }
            writer.end();
        }
        infoMessages.clear();

        _resources.renderInformalParameters(writer);
    }

    void afterRender(MarkupWriter writer) {

    }

    /**
     * Ajout d'un message d'info.
     *
     * @param message le message
     */
    public void addInfo(String message) {
        infoMessages.add(message);
    }

    /**
     * Ajout d'un message d'erreur.
     *
     * @param message le message
     */
    public void addError(String message) {
        errorMessages.add(message);
    }
}