/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: VoteForPoll.java 3674 2012-09-04 16:00:18Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/vote/VoteForPoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll.vote;

import com.google.common.base.Preconditions;
import com.opensymphony.xwork2.Preparable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.services.impl.VoteService;
import org.chorem.pollen.votecounting.strategy.VoteCountingStrategy;

import static org.nuiton.i18n.I18n.n_;

/**
 * Votes to a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @since 1.3
 */
public class VoteForPoll extends AbstractVoteAction implements Preparable {

    private static final Log log = LogFactory.getLog(VoteForPoll.class);

    private static final long serialVersionUID = 1L;

    @Override
    public boolean isModerate() {
        return false;
    }

    @Override
    public void prepare() throws Exception {

        prepareVotePage();

        // Messages
        Poll poll = getPoll();

        if (poll.isClosed()) {
            addFlashWarning(_("pollen.information.pollClosed"));
        } else if (!isPollStarted()) {
            addFlashWarning(_("pollen.information.pollNotStarted"));
        } else if (isPollFinished()) {
            addFlashWarning(_("pollen.information.pollFinished"));
        } else if (!isVoteAllowed()) {

            addFlashWarning(_("pollen.information.pollCanNotVote"));
        }

        if (isPollChoiceRunning()) {
            addFlashMessage(_("pollen.information.pollChoiceRunning"));
        }
    }

    @Override
    public void validate() {

        Vote vote = getVote();
        Preconditions.checkNotNull(vote);

        PollAccount pollAccount = getVote().getPollAccount();
        Preconditions.checkNotNull(pollAccount);

        Poll poll = getPoll();

        if (!vote.isAnonymous()) {

            String name = pollAccount.getVotingId();
            if (StringUtils.isBlank(name)) {
                addFieldError("pollAccount.votingId",
                              _("pollen.error.pollAccount.votingId.required"));
            }

            // check if the votingId is available
            if (!getVoteService().isVotingIdFree(poll, pollAccount.getTopiaId(),name)) {
                addFieldError("pollAccount.votingId",
                              _("pollen.error.user.alreadyVoted", name));
            }
        }

        int nbVotes = 0;
        int totalValues = 0;

        VoteCountingStrategy strategy = getVoteCountingStrategy(getPoll());

        boolean voteValid = true;
        int voteNumber = 0;
        for (VoteToChoice voteToChoice : vote.getChoiceVoteToChoice()) {

            Integer value = voteToChoice.getVoteValue();

            if (value==null) {

                // check value if same as the one return by request
                // otherwise it means there is a conversion error
                String[] values = getParameters().get("vote.choiceVoteToChoice[" + voteNumber + "].voteValue");
                String paramValue = values==null || values.length<1?null:values[0];

                if (StringUtils.isNotBlank(paramValue)) {

                    // vote value changed
                    String validMessage =
                            _("pollen.error.vote.invalidFormat", paramValue);
                    addFieldError("vote.choices", validMessage);
                    voteValid = false;
                    break;
                }
            }
            // check if vote is null ?
            boolean voteNull = strategy.isVoteValueNull(value);

            if (voteNull) {
                // null vote, can skip his validation
                continue;
            }

            // check if vote value is valid ?
            boolean valid = strategy.isVoteValueValid(value);

            if (valid) {

                // ok this vote value is valid, use it
                nbVotes++;
                totalValues += value;
            } else {

                // not a valid vote value, mark it and skip other fields validation
                String validMessage =
                        strategy.getVoteValueNotValidMessage(getLocale());
                addFieldError("vote.choices", validMessage);
                voteValid = false;
                break;
            }
        }
        if (voteValid) {

            // check that the number of choices is not greater than maxChoiceNb
            int maxChoice = getPoll().getMaxChoiceNb();
            if (maxChoice != 0 && nbVotes > maxChoice) {
                addFieldError("vote.choices",
                              _("pollen.error.vote.maxChoiceNb", maxChoice));
            }

            // check that total vote value is ok
            if (!strategy.isTotalVoteValueValid(totalValues)) {

                // not valid
                String errorMessage =
                        strategy.getTotalVoteValueNotValidMessage(getLocale());
                addFieldError("vote.choices", errorMessage);
            }
        }
    }

    @Override
    public String execute() throws Exception {

        // REMARQUES :
        // - pas le droit de modif si le pollAccount est rattaché a un userAccount et qu'on est pas loggé ?!?
        // - Ce serait plus simple que l'Admin ne puisse jamais voter, il ne peut que modérer les votes
        //TODO tchemit-2012-08-12 Non l'admin est un user comme les autres,
        //TODO on doit pouvoir gérer tous les cas de manière simple (enfin je crois)

        Poll poll = getPoll();

        Vote vote = getVote();

        VoteService voteService = getVoteService();

        boolean newVote = StringUtils.isBlank(vote.getTopiaId());

        Vote savedVote;
        if (newVote) {

            // create a new vote
            savedVote = voteService.createVote(poll, vote);
        } else {

            // update existing vote
            savedVote = voteService.updateVote(poll, vote);
        }

        clearFlashMessages();

        if (StringUtils.isBlank(getUserSecurityContext().getAccountId())) {

            // no accountId in url, always propose it (for connected
            // user to be able to bookmark their votes)

            String pollId = poll.getPollId();
            String accountId = savedVote.getPollAccount().getAccountId();

            PollUrl updateUrl = getPollUrlService().getPollVoteUrl(
                    poll, savedVote.getPollAccount());

            if (log.isDebugEnabled()) {
                log.debug(String.format(
                        "UpdateURL for poll '%s' and account '%s' = %s",
                        pollId, accountId, updateUrl));
            }

            String message;

            boolean connected = getUserSecurityContext().isConnected();

            if (connected) {
                message = n_("pollen.information.vote.createdWithUpdateUrl.notConnected");
            } else {
                message = n_("pollen.information.vote.createdWithUpdateUrl");
            }

            addFlashMessage(_(message, updateUrl));

        } else {

            // there is a accountId in url, just inform vote is saved
            addFlashMessage(_("pollen.information.vote.created"));
        }

        return SUCCESS;
    }
}
