/*
 * #%L
 * Pollen :: Persistence
 * 
 * $Id: PollenTopiaRootContextFactory.java 3257 2012-04-13 11:28:02Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1/pollen-persistence/src/main/java/org/chorem/pollen/PollenTopiaRootContextFactory.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen;

import com.google.common.collect.Sets;
import com.google.common.io.Closeables;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.TopiaRuntimeException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenTopiaRootContextFactory {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenTopiaRootContextFactory.class);

    public TopiaContext newEmbeddedDatabase(File dir, String configPath) {

        File databaseFile = new File(dir, "db");

        String databaseAbsolutePath = databaseFile.getAbsolutePath();

        // prepare call to topia-context factory
        Properties properties = new Properties();
        InputStream input = null;
        try {
            input = getClass().getResourceAsStream(configPath);
            properties.load(input);
            input.close();
        } catch (IOException e) {
            throw new PollenTechnicalException(e);
        } finally {
            Closeables.closeQuietly(input);
        }
        properties.setProperty(
                TopiaContextFactory.CONFIG_URL,
                "jdbc:h2:file:" + databaseAbsolutePath + "/pollen");

        // add entities to the context
        properties.setProperty(
                TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES,
                PollenDAOHelper.getImplementationClassesAsString());

        TopiaContext rootContext;
        try {
            rootContext = TopiaContextFactory.getContext(properties);
        } catch (TopiaNotFoundException e) {
            throw new TopiaRuntimeException(e);
        }

        if (log.isDebugEnabled()) {
            log.debug("will output database in " + databaseAbsolutePath);
        }

        return rootContext;
    }

    public TopiaContext newDatabaseFromConfig(PollenConfiguration config) {

        Properties properties = config.getProperties();

        return newDatabaseFromProperties(properties);
    }

    public TopiaContext newDatabaseFromProperties(Properties properties) {

        if (log.isDebugEnabled()) {
            log.debug("Database settings are :");
            Set<String> keysToDisplay = Sets.newHashSet(
                    "hibernate.dialect",
                    "hibernate.connection.driver_class",
                    "hibernate.connection.url",
                    "hibernate.connection.username");
            for (String key : keysToDisplay) {
                log.debug(String.format("%s=%s", key, properties.getProperty(key)));
            }
        }

        // add entities to the context
        String classesKey = TopiaContextFactory.CONFIG_PERSISTENCE_CLASSES;
        String classesValue = PollenDAOHelper.getImplementationClassesAsString();
        properties.setProperty(classesKey, classesValue);

        TopiaContext rootContext;
        try {
            rootContext = TopiaContextFactory.getContext(properties);
        } catch (TopiaNotFoundException e) {
            throw new TopiaRuntimeException(e);
        }

        return rootContext;
    }
}
