/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.converters;

import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.business.services.ServicePollAccountImpl;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.CommentDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;

/**
 * Classe de conversion d'entités en DTO pour le module voteCounting.
 *
 * @author fdesbois
 * @author rannou
 * @version $Id: DataVoteCountingConverter.java 2696 2009-08-10 16:18:15Z
 *          nrannou $
 */
public class DataVoteCountingConverter extends DataConverter {

    /**
     * Création d'un DTO sondage d'export à partir d'une entité sondage.
     *
     * @param ePoll l'entité sondage
     * @return le DTO sondage d'export
     */
    public static PollDTO createPollDTOForVoteCounting(Poll ePoll) {
        PollDTO pollDTO = new PollDTO(ePoll.getPollId());
        pollDTO.setTitle(ePoll.getTitle());
        pollDTO.setDescription(ePoll.getDescription());
        pollDTO.setBeginChoiceDate(ePoll.getBeginChoiceDate());
        pollDTO.setBeginDate(ePoll.getBeginDate());
        pollDTO.setEndDate(ePoll.getEndDate());
        pollDTO.setAnonymous(ePoll.getAnonymous());
        pollDTO.setAnonymousVoteAllowed(ePoll.getAnonymousVoteAllowed());
        pollDTO.setPublicResults(ePoll.getPublicResults());
        pollDTO.setContinuousResults(ePoll.getContinuousResults());
        pollDTO.setChoiceAddAllowed(ePoll.getChoiceAddAllowed());
        pollDTO.setClosed(ePoll.getClosed());
        pollDTO.setCreatorId(ePoll.getCreator().getVotingId());
        pollDTO.setCreatorEmail(ePoll.getCreator().getEmail());
        pollDTO.setMaxChoiceNb(ePoll.getMaxChoiceNb());
        pollDTO.setPollType(EnumController.getPollType(ePoll));
        pollDTO.setChoiceType(EnumController.getChoiceType(ePoll));
        pollDTO.setVoteCounting(EnumController.getVoteCountingType(ePoll));

        for (Comment comment : ePoll.getComment()) {
            pollDTO.getComments().add(createPollCommentDTO(comment));
        }
        for (Choice choice : ePoll.getChoice()) {
            pollDTO.getChoices().add(createPollChoiceDTO(choice));
        }
        if (ePoll.getVotingList() != null && !ePoll.getVotingList().isEmpty()) {
            for (VotingList list : ePoll.getVotingList()) {
                pollDTO.getVotingGroups().add(createVotingGroupDTO(list));
            }
        } else {
            // un groupe par défaut si il y en a pas
            VotingGroupDTO group = new VotingGroupDTO("unique", 1);
            group.setName("unique");
            for (Vote vote : ePoll.getVote()) {
                group.getVotingPersons().add(createVotingPersonDTO(vote));
            }
            pollDTO.getVotingGroups().add(group);
        }
        return pollDTO;
    }

    private static CommentDTO createPollCommentDTO(Comment comment) {
        CommentDTO commentDTO = new CommentDTO(comment.getPollAccount()
                .getVotingId(), comment.getText());
        return commentDTO;
    }

    private static PollChoiceDTO createPollChoiceDTO(Choice choice) {
        PollChoiceDTO choiceDTO = new PollChoiceDTO(choice.getTopiaId());
        String choiceName = choice.getName();
        choiceDTO.setName(choiceName);
        choiceDTO.setHidden(choiceName != null &&
                choiceName.startsWith(NumberMethod.HIDDEN_PREFIX));
        choiceDTO.setDescription(choice.getDescription());
        return choiceDTO;
    }

    private static VotingGroupDTO createVotingGroupDTO(VotingList list) {
        VotingGroupDTO group = new VotingGroupDTO(list.getTopiaId(), list
                .getWeight());
        group.setName(list.getName());

        for (PersonToList pToList : list.getPollAccountPersonToList()) {
            Vote vote = getPollVoteByPollAccount(list.getPoll(), pToList
                    .getPollAccount());
            if (vote != null) { // Pas de vote pour cette personne : doit engendrer erreur ?!?
                group.getVotingPersons().add(createVotingPersonDTO(vote));
            }
        }
        return group;
    }

    private static VotingPersonDTO createVotingPersonDTO(Vote vote) {
        VotingPersonDTO votingPerson = new VotingPersonDTO(vote
                .getPollAccount().getVotingId(), vote.getWeight());
        votingPerson.setEmail(vote.getPollAccount().getEmail());

        for (VoteToChoice vToChoice : vote.getChoiceVoteToChoice()) {
            if (vToChoice != null && vToChoice.getChoice() != null) {
                votingPerson.getChoices().add(createVoteToChoiceDTO(vToChoice));
            }
        }
        return votingPerson;
    }

    private static VoteToChoiceDTO createVoteToChoiceDTO(VoteToChoice vToChoice) {
        String topiaId = vToChoice.getChoice().getTopiaId();
        VoteToChoiceDTO vote = new VoteToChoiceDTO(topiaId, vToChoice
                .getVoteValue());

        return vote;
    }

    /* A Deplacer au bon endroit (PollImpl) si c'est possible */
    private static Vote getPollVoteByPollAccount(Poll poll, PollAccount account) {
        for (Vote vote : poll.getVote()) {
            if (account.getTopiaId().equals(vote.getPollAccount().getTopiaId())) {
                return vote;
            }
        }
        return null;
    }

    /**
     * Création d'un DTO sondage à partir d'un DTO sondage d'export.
     *
     * @param pollDTO le DTO sondage d'export
     * @return le DTO sondage
     */
    public static org.chorem.pollen.business.dto.PollDTO createPollDTO(
            PollDTO pollDTO) {
        org.chorem.pollen.business.dto.PollDTO bPollDTO = new org.chorem.pollen.business.dto.PollDTO(
                pollDTO.getPollId());
        bPollDTO.setTitle(pollDTO.getTitle());
        bPollDTO.setDescription(pollDTO.getDescription());
        bPollDTO.setBeginChoiceDate(pollDTO.getBeginChoiceDate());
        bPollDTO.setBeginDate(pollDTO.getBeginDate());
        bPollDTO.setEndDate(pollDTO.getEndDate());
        bPollDTO.setAnonymous(pollDTO.isAnonymous());
        bPollDTO.setAnonymousVoteAllowed(pollDTO.isAnonymousVoteAllowed());
        bPollDTO.setPublicResults(pollDTO.isPublicResults());
        bPollDTO.setContinuousResults(pollDTO.isContinuousResults());
        bPollDTO.setChoiceAddAllowed(pollDTO.isChoiceAddAllowed());
        bPollDTO.setClosed(pollDTO.isClosed());
        bPollDTO.setCreatorId(pollDTO.getCreatorId());
        bPollDTO.setCreatorEmail(pollDTO.getCreatorEmail());
        bPollDTO.setMaxChoiceNb(pollDTO.getMaxChoiceNb());
        bPollDTO.setPollType(pollDTO.getPollType());
        bPollDTO.setChoiceType(pollDTO.getChoiceType());
        bPollDTO.setVoteCounting(pollDTO.getVoteCounting());

        for (CommentDTO comment : pollDTO.getComments()) {
            bPollDTO.getCommentDTOs().add(createPollCommentDTO(comment));
        }
        for (PollChoiceDTO choice : pollDTO.getChoices()) {
            bPollDTO.getChoiceDTOs().add(createPollChoiceDTO(choice));
        }
        // TODO conversion des groupes et des votes

        return bPollDTO;
    }

    private static org.chorem.pollen.business.dto.CommentDTO createPollCommentDTO(
            CommentDTO commentDTO) {
        ServicePollAccount spa = new ServicePollAccountImpl();
        PollAccount account = spa.createPollAccount(commentDTO.getVotingID(),
                null, null);

        org.chorem.pollen.business.dto.CommentDTO bCommentDTO = new org.chorem.pollen.business.dto.CommentDTO(
                commentDTO.getText(), null, account.getTopiaId());
        return bCommentDTO;
    }

    private static org.chorem.pollen.business.dto.ChoiceDTO createPollChoiceDTO(
            PollChoiceDTO pollChoiceDTO) {
        org.chorem.pollen.business.dto.ChoiceDTO bChoiceDTO = new org.chorem.pollen.business.dto.ChoiceDTO();
        String pollChoiceName = pollChoiceDTO.getName();
        bChoiceDTO.setName(pollChoiceName);
        bChoiceDTO.setHidden(pollChoiceName != null
                    && pollChoiceName.startsWith(NumberMethod.HIDDEN_PREFIX));
        bChoiceDTO.setDescription(pollChoiceDTO.getDescription());
        return bChoiceDTO;
    }

}
