/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.converters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.nuiton.topia.TopiaException;

/**
 * Gestion de la conversion des dtos Choice en entité et vice-versa.
 *
 * @author enema
 * @version $Id: DataChoiceConverter.java 2859 2010-02-09 13:50:24Z jruchaud $
 */
public class DataChoiceConverter extends DataConverter {

    /**
     * Création d'une entité choix à partir d'un dto choix.
     *
     * @param choiceDTO Le dto choix
     * @param eChoice L'entité choix
     */
    public void populateChoiceEntity(ChoiceDTO choiceDTO, Choice eChoice)
            throws TopiaException {
        eChoice.setName(choiceDTO.getName());
        eChoice.setDescription(choiceDTO.getDescription());
        eChoice.setValidate(choiceDTO.isValidate());

        if (choiceDTO.getPollId().length() > 0) {
            PollDAO pollDAO = PollenModelDAOHelper
                    .getPollDAO(super.transaction);
            Poll pollEntity = pollDAO.findByTopiaId(choiceDTO.getPollId());
            eChoice.setPoll(pollEntity);
        }
    }

    /**
     * Création d'un dto choix à partir d'une entité choix.
     *
     * @param eChoice L'entité choix
     * @return Le dto choix
     */
    public ChoiceDTO createChoiceDTO(Choice eChoice) {
        ChoiceDTO choiceDTO = new ChoiceDTO();
        choiceDTO.setId(eChoice.getTopiaId());
        String eChoiceName = eChoice.getName();
        choiceDTO.setName(eChoiceName);
        choiceDTO.setHidden(eChoiceName != null
                    && eChoiceName.startsWith(NumberMethod.HIDDEN_PREFIX));
        choiceDTO.setDescription(eChoice.getDescription());
        choiceDTO.setValidate(eChoice.getValidate());

        if (eChoice.getPoll() != null) {
            choiceDTO.setPollId(eChoice.getPoll().getTopiaId());
        }

        return choiceDTO;
    }

    /**
     * Retourne une liste de dtos choix à partir d'une collection d'entités.
     *
     * @param cChoices Les entités choix
     * @return la liste des dtos choix
     */
    public List<ChoiceDTO> createChoiceDTOs(Collection<Choice> cChoices) {
        List<ChoiceDTO> results = new ArrayList<ChoiceDTO>();
        ChoiceDTO dto;
        for (Choice c : cChoices) {
            dto = this.createChoiceDTO(c);
            results.add(dto);
        }
        return results;
    }
}
