/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataResultConverter;
import org.chorem.pollen.business.converters.DataVoteCountingConverter;
import org.chorem.pollen.business.converters.EnumController;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.dto.ResultListDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Result;
import org.chorem.pollen.business.utils.ContextUtil;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.PollExportDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.services.ServiceExport;
import org.chorem.pollen.votecounting.services.ServiceExportImpl;
import org.chorem.pollen.votecounting.services.ServiceVoteCounting;
import org.chorem.pollen.votecounting.services.ServiceVoteCountingImpl;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des résultats.
 *
 * @author fdesbois
 * @author rannou
 * @version $Id: ServiceResultsImpl.java 2836 2010-01-11 12:46:59Z schorlet $
 */
public class ServiceResultsImpl implements ServiceResults {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceResultsImpl.class);

    public ServiceResultsImpl() {
    }

    @Override
    public ResultListDTO getAllResults(String pollId) {
        return getResults(pollId, null, false, false);
    }

    @Override
    public ResultListDTO getResultsByVoteCounting(String pollId,
            VoteCountingType voteCounting) {
        return getResults(pollId, voteCounting, false, false);
    }

    @Override
    public ResultListDTO getGroupResults(String pollId) {
        return getResults(pollId, null, true, true);
    }

    @Override
    public ResultListDTO getNormalResults(String pollId) {
        return getResults(pollId, null, true, false);
    }

    /**
     * Récupère les résultats d'un sondage.
     *
     * @param pollId le sondage
     * @param voteCounting type de dépouillement
     * @param byGroup résultats par groupe
     * @param groupOnly résultats uniquement par groupe
     * @return les résultats du sondage
     */
    private ResultListDTO getResults(String pollId,
            VoteCountingType voteCounting, boolean byGroup, boolean groupOnly) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            PollDAO daoPoll = PollenModelDAOHelper.getPollDAO(transaction);
            Poll ePoll = daoPoll.findByPollId(pollId);

            DataResultConverter converter = new DataResultConverter(transaction);
            if (log.isDebugEnabled()) {
                log.debug(ePoll.getPollId() + " (" + voteCounting
                        + ") has results: " + hasResults(ePoll, voteCounting));
            }

            // Dépouillement du sondage.
            // S'il n'existe pas de résultats ou si le sondage est encore ouvert.
            //if (!hasResults(ePoll, voteCounting) || !ePoll.getIsClosed()) {
            PollDTO dto = DataVoteCountingConverter
                    .createPollDTOForVoteCounting(ePoll);

            if (voteCounting != null) {
                dto.setVoteCounting(voteCounting);
            }

            ePoll.clearResult();

            ServiceVoteCounting service = new ServiceVoteCountingImpl();
            VoteCountingResultDTO result = service.executeVoteCounting(dto);
            converter.populateResultEntities(result);
            if (EnumController.isGroupType(ePoll)) {
                result = service.executeGroupCounting(dto);
                converter.populateResultEntities(result);
            }

            //}

            transaction.commitTransaction();

            // Conversion et trie des résultats
            List<ResultDTO> list = converter.createResultDTOs(ePoll);
            transaction.closeContext();

            VoteCountingType voteCountingType = EnumController
                    .getVoteCountingType(ePoll);
            
            Iterator<ResultDTO> it = list.iterator();
            while (it.hasNext()) {
                ResultDTO curr = it.next();

                // Cas d'un dépouillement particulier
                // Suppression des resultats qui ne sont pas de se dépouillement
                if (voteCounting != null
                        && curr.getVoteCounting() != voteCounting) {
                    it.remove();
                }

                // Cas d'un sondage de type GROUP avec filtre
                if (byGroup) {

                    // Filtre group : Suppression resultats non group
                    if (groupOnly && !curr.isByGroup()) {
                        it.remove();
                    }

                    // Filtre non group : Suppression resultats group
                    else if (!groupOnly && curr.isByGroup()) {
                        it.remove();
                    }

                    // Suppression des autres resultats de depouillements differents
                    if (curr.getVoteCounting() != voteCountingType) {
                        it.remove();
                    }
                }
            }

            ResultListDTO resultListDTO = new ResultListDTO();
            resultListDTO.setResultDTOs(list);
            resultListDTO.setVoteCountingResultDTO(result);
            
            return resultListDTO;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    /**
     * Retourne vrai si le sondage a des résultats.
     *
     * @param ePoll le sondage
     * @param type type de dépouillement
     * @return vrai si le sondage a des résultats
     */
    private boolean hasResults(Poll ePoll, VoteCountingType type) {
        Collection<Result> results = ePoll.getResult();
        if (results == null || results.isEmpty()) {
            return false;
        }
        if (!results.isEmpty() && type == null) {
            return true;
        }

        for (Result curr : results) {
            if (EnumController.getVoteCountingType(curr) == type) {
                return true;
            }
        }
        return false;
    }

    @Override
    public String exportPoll(String pollId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            // Recherche du sondage
            PollDAO daoPoll = PollenModelDAOHelper.getPollDAO(transaction);
            Poll ePoll = daoPoll.findByPollId(pollId);

            // Dépouillement du sondage.
            DataResultConverter converter = new DataResultConverter(transaction);
            PollDTO dto = DataVoteCountingConverter
                    .createPollDTOForVoteCounting(ePoll);

            ePoll.clearResult();

            ServiceVoteCounting service = new ServiceVoteCountingImpl();
            List<VoteCountingResultDTO> results = new ArrayList<VoteCountingResultDTO>();
            VoteCountingResultDTO result = service.executeVoteCounting(dto);
            converter.populateResultEntities(result);
            results.add(result);

            if (EnumController.isGroupType(ePoll)) {
                VoteCountingResultDTO groupResult = service
                        .executeGroupCounting(dto);
                converter.populateResultEntities(groupResult);
                results.add(groupResult);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            // Transformation du sondage
            PollExportDTO pollExport = new PollExportDTO();
            pollExport.setPollId(pollId);
            pollExport.setPoll(dto);
            pollExport.setVoteCountingResults(results);

            // Export du sondage
            ServiceExport serviceExport = new ServiceExportImpl();
            String filename = serviceExport.executeExport(pollExport);

            if (log.isInfoEnabled()) {
                log.info("Poll exported: " + pollId);
            }

            return filename;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public String importPoll(String filePath) {
        return importPoll(filePath, null);
    }

    @Override
    public String importPoll(String filePath, UserDTO user) {
        String topiaId = "";

        // Import du sondage
        ServiceExport serviceExport = new ServiceExportImpl();
        PollExportDTO pollExportDTO = serviceExport.executeImport(filePath);

        // Transformation du sondage
        // FIXME do not call a Service from an other one
        ServicePoll servicePoll = new ServicePollImpl();
        org.chorem.pollen.business.dto.PollDTO poll = DataVoteCountingConverter
                .createPollDTO(pollExportDTO.getPoll());

        // Enregistrement du sondage
        if (user != null) {
            poll.setUserId(user.getId());
        }
        topiaId = servicePoll.createPoll(poll);

        if (log.isInfoEnabled()) {
            log.info("Poll imported: " + topiaId);
        }

        return topiaId;
    }
}