/*
 * Decompiled with CFR 0.152.
 */
package nl.knaw.dans.common.dbflib;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import nl.knaw.dans.common.dbflib.BooleanValue;
import nl.knaw.dans.common.dbflib.ByteArrayValue;
import nl.knaw.dans.common.dbflib.CorruptedTableException;
import nl.knaw.dans.common.dbflib.DateValue;
import nl.knaw.dans.common.dbflib.DbfHeader;
import nl.knaw.dans.common.dbflib.DbfLibException;
import nl.knaw.dans.common.dbflib.Field;
import nl.knaw.dans.common.dbflib.IfNonExistent;
import nl.knaw.dans.common.dbflib.InvalidFieldLengthException;
import nl.knaw.dans.common.dbflib.InvalidFieldTypeException;
import nl.knaw.dans.common.dbflib.Memo;
import nl.knaw.dans.common.dbflib.NumberValue;
import nl.knaw.dans.common.dbflib.Record;
import nl.knaw.dans.common.dbflib.RecordTooLargeException;
import nl.knaw.dans.common.dbflib.StringValue;
import nl.knaw.dans.common.dbflib.Type;
import nl.knaw.dans.common.dbflib.Util;
import nl.knaw.dans.common.dbflib.Value;
import nl.knaw.dans.common.dbflib.Version;

public class Table {
    private static final int MARKER_RECORD_DELETED = 42;
    private static final int MARKER_EOF = 26;
    private static final int MARKER_RECORD_VALID = 32;
    private final File tableFile;
    private final DbfHeader header = new DbfHeader();
    private final String charsetName;
    private Memo memo = null;
    private RandomAccessFile raFile = null;

    public Table(File tableFile) throws IllegalArgumentException {
        this(tableFile, Charset.defaultCharset().name());
    }

    public Table(File tableFile, String charsetName) throws IllegalArgumentException {
        if (tableFile == null) {
            throw new IllegalArgumentException("Table file must not be null");
        }
        this.tableFile = tableFile;
        this.charsetName = charsetName == null ? Charset.defaultCharset().name() : charsetName;
        Charset.forName(this.charsetName);
    }

    public Table(File tableFile, Version version, List<Field> fields, String charsetName) throws InvalidFieldTypeException, InvalidFieldLengthException {
        this(tableFile, charsetName);
        this.header.setVersion(version);
        this.header.setHasMemo(Table.hasMemo(fields));
        this.header.setFields(fields);
    }

    public Table(File tableFile, Version version, List<Field> fields) throws InvalidFieldTypeException, InvalidFieldLengthException {
        this(tableFile, version, fields, Charset.defaultCharset().name());
    }

    private static boolean hasMemo(List<Field> fields) {
        for (Field field : fields) {
            if (field.getType() != Type.MEMO) continue;
            return true;
        }
        return false;
    }

    public void open() throws IOException, CorruptedTableException {
        this.open(IfNonExistent.ERROR);
    }

    public void open(IfNonExistent ifNonExistent) throws IOException, CorruptedTableException {
        if (this.tableFile.exists()) {
            this.raFile = new RandomAccessFile(this.tableFile, "rw");
            this.header.readAll(this.raFile);
        } else if (ifNonExistent.isCreate()) {
            this.raFile = new RandomAccessFile(this.tableFile, "rw");
            this.header.writeAll(this.raFile);
        } else if (ifNonExistent.isError()) {
            throw new FileNotFoundException("Input file " + this.tableFile + " not found");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close() throws IOException {
        try {
            if (this.raFile != null) {
                this.raFile.close();
            }
        }
        finally {
            this.raFile = null;
            this.ensureMemoClosed();
        }
    }

    public void delete() throws IOException {
        this.close();
        this.tableFile.delete();
        if (this.memo != null) {
            this.memo.delete();
        }
    }

    public Date getLastModifiedDate() {
        this.checkOpen();
        return this.header.getLastModifiedDate();
    }

    public String getName() {
        return this.tableFile.getName();
    }

    public List<Field> getFields() {
        this.checkOpen();
        return this.header.getFields();
    }

    public Iterator<Record> recordIterator() {
        return this.recordIterator(false);
    }

    public Iterator<Record> recordIterator(boolean includeDeleted) {
        return new RecordIterator(includeDeleted);
    }

    public void addRecord(Object ... fieldValues) throws IOException, DbfLibException {
        if (fieldValues.length > this.header.getFields().size()) {
            throw new RecordTooLargeException("Trying to add " + fieldValues.length + " fields while there are only " + this.header.getFields().size() + " defined in the table file");
        }
        HashMap<String, Value> map = new HashMap<String, Value>();
        Iterator<Field> fieldIterator = this.header.getFields().iterator();
        for (Object fieldValue : fieldValues) {
            Field field = fieldIterator.next();
            map.put(field.getName(), this.createValueObject(fieldValue));
        }
        this.addRecord(new Record(map));
    }

    private Value createValueObject(Object value) {
        if (value instanceof Number) {
            return new NumberValue((Number)value);
        }
        if (value instanceof String) {
            return new StringValue((String)value, this.charsetName);
        }
        if (value instanceof Boolean) {
            return new BooleanValue((Boolean)value);
        }
        if (value instanceof Date) {
            return new DateValue((Date)value);
        }
        if (value instanceof byte[]) {
            return new ByteArrayValue((byte[])value);
        }
        return null;
    }

    public void addRecord(Record record) throws IOException, DbfLibException {
        this.updateRecordAt(this.header.getRecordCount(), record);
        this.raFile.writeByte(26);
        this.writeRecordCount(this.header.getRecordCount() + 1);
    }

    public void updateRecordAt(int index, Record record) throws IOException, DbfLibException {
        this.checkOpen();
        this.jumpToRecordAt(index);
        this.raFile.writeByte(32);
        for (Field field : this.header.getFields()) {
            byte[] raw = record.getRawValue(field);
            if (raw == null) {
                raw = Util.repeat((byte)32, field.getLength());
            } else if (field.getType() == Type.MEMO || field.getType() == Type.BINARY || field.getType() == Type.GENERAL) {
                int i = this.writeMemo(raw);
                raw = this.header.getVersion() == Version.DBASE_4 || this.header.getVersion() == Version.DBASE_5 ? String.format("%0" + field.getLength() + "d", i).getBytes() : String.format("%" + field.getLength() + "d", i).getBytes();
            }
            this.raFile.write(raw);
            if (raw.length >= field.getLength()) continue;
            this.raFile.write(Util.repeat((byte)0, field.getLength() - raw.length));
        }
    }

    public void deleteRecordAt(int index) throws IOException {
        this.checkOpen();
        this.jumpToRecordAt(index);
        this.raFile.writeByte(42);
    }

    private int writeMemo(byte[] memoText) throws IOException, CorruptedTableException {
        this.ensureMemoOpened(IfNonExistent.CREATE);
        return this.memo.writeMemo(memoText);
    }

    private void writeRecordCount(int recordCount) throws IOException {
        this.raFile.seek(4L);
        this.header.setRecordCount(recordCount);
        this.header.writeRecordCount(this.raFile);
    }

    private void checkOpen() {
        if (this.raFile == null) {
            throw new IllegalStateException("Table should be open for this operation");
        }
    }

    private byte[] readMemo(String memoIndex) throws IOException, CorruptedTableException {
        this.ensureMemoOpened(IfNonExistent.ERROR);
        if (memoIndex.trim().isEmpty()) {
            return null;
        }
        return this.memo.readMemo(Integer.parseInt(memoIndex.trim()));
    }

    private void ensureMemoOpened(IfNonExistent ifNonExistent) throws IOException, CorruptedTableException {
        if (this.memo != null) {
            return;
        }
        this.openMemo(ifNonExistent);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureMemoClosed() throws IOException {
        if (this.memo != null) {
            try {
                this.memo.close();
            }
            finally {
                this.memo = null;
            }
        }
    }

    private void openMemo(IfNonExistent ifNonExistent) throws IOException, CorruptedTableException {
        File memoFile = Util.getMemoFile(this.tableFile, this.header.getVersion());
        if (memoFile == null) {
            String extension;
            String string = extension = this.header.getVersion() == Version.FOXPRO_26 ? ".fpt" : ".dbt";
            if (ifNonExistent.isError()) {
                throw new CorruptedTableException("Could not find file '" + Util.stripExtension(this.tableFile.getPath()) + extension + "' (or multiple matches for the file)");
            }
            if (ifNonExistent.isCreate()) {
                String tableFilePath = this.tableFile.getPath();
                memoFile = new File(tableFilePath.substring(0, tableFilePath.length() - ".dbf".length()) + extension);
            } else assert (false) : "Programming error: cannot ignore non existing memo.";
        }
        this.memo = new Memo(memoFile, this.header.getVersion());
        this.memo.open(ifNonExistent);
    }

    public Record getRecordAt(int index) throws IOException, CorruptedTableException {
        this.checkOpen();
        if (index >= this.header.getRecordCount()) {
            throw new NoSuchElementException(String.format("Invalid index: %d", index));
        }
        this.jumpToRecordAt(index);
        byte firstByteOfRecord = this.raFile.readByte();
        if (firstByteOfRecord == 26) {
            throw new NoSuchElementException(String.format("Invalid index: %d", index));
        }
        HashMap<String, Value> recordValues = new HashMap<String, Value>();
        block8: for (Field field : this.header.getFields()) {
            byte[] rawData = Util.readStringBytes(this.raFile, field.getLength());
            switch (field.getType()) {
                case NUMBER: 
                case FLOAT: {
                    recordValues.put(field.getName(), new NumberValue(field, rawData));
                    continue block8;
                }
                case CHARACTER: {
                    recordValues.put(field.getName(), new StringValue(field, rawData, this.charsetName));
                    continue block8;
                }
                case LOGICAL: {
                    recordValues.put(field.getName(), new BooleanValue(field, rawData));
                    continue block8;
                }
                case DATE: {
                    recordValues.put(field.getName(), new DateValue(field, rawData));
                    continue block8;
                }
                case MEMO: {
                    byte[] memoTextBytes = this.readMemo(new String(rawData));
                    recordValues.put(field.getName(), memoTextBytes == null ? null : new StringValue(field, memoTextBytes, this.charsetName));
                    continue block8;
                }
                case GENERAL: 
                case BINARY: 
                case PICTURE: {
                    recordValues.put(field.getName(), new ByteArrayValue(this.readMemo(new String(rawData))));
                    continue block8;
                }
            }
            throw new RuntimeException("Not all types handled");
        }
        return new Record(firstByteOfRecord == 42, recordValues);
    }

    public void pack() throws IOException, DbfLibException {
        Iterator<Record> iterator = this.recordIterator(false);
        int i = 0;
        while (iterator.hasNext()) {
            this.updateRecordAt(i++, iterator.next());
        }
        this.writeRecordCount(i);
        this.jumpToRecordAt(i);
        this.raFile.write(26);
        this.raFile.setLength(this.raFile.getFilePointer());
    }

    public String getCharsetName() {
        return this.charsetName;
    }

    public Version getVersion() {
        return this.header.getVersion();
    }

    private void jumpToRecordAt(int index) throws IOException {
        this.raFile.seek(this.header.getLength() + index * this.header.getRecordLength());
    }

    public int getRecordCount() {
        return this.header.getRecordCount();
    }

    private class RecordIterator
    implements Iterator<Record> {
        private final boolean includeDeleted;
        private int recordCounter = -1;
        private boolean currentElementDeleted = false;

        RecordIterator(boolean includeDeleted) {
            this.includeDeleted = includeDeleted;
        }

        @Override
        public boolean hasNext() {
            try {
                return this.recordCounter + 1 < Table.this.header.getRecordCount() && (this.includeDeleted || !this.followingRecordsAreAllDeleted());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        private boolean followingRecordsAreAllDeleted() throws IOException {
            byte b;
            int index = this.recordCounter + 1;
            do {
                Table.this.jumpToRecordAt(index++);
                b = Table.this.raFile.readByte();
                if (b != 32) continue;
                return false;
            } while (index < Table.this.header.getRecordCount() && b == 42);
            return true;
        }

        @Override
        public Record next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            Table.this.checkOpen();
            try {
                Record record;
                do {
                    record = Table.this.getRecordAt(++this.recordCounter);
                } while (!this.includeDeleted && record.isMarkedDeleted());
                this.currentElementDeleted = false;
                return record;
            }
            catch (IOException ioException) {
                throw new RuntimeException(ioException.getMessage(), ioException);
            }
            catch (CorruptedTableException corruptedTableException) {
                throw new RuntimeException(corruptedTableException.getMessage(), corruptedTableException);
            }
        }

        @Override
        public void remove() {
            if (this.recordCounter == 0 || this.recordCounter >= Table.this.header.getRecordCount()) {
                throw new NoSuchElementException();
            }
            if (this.currentElementDeleted) {
                throw new RuntimeException("Current element already removed");
            }
            try {
                Table.this.deleteRecordAt(this.recordCounter);
                this.currentElementDeleted = true;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}

