/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.ldap.sdk;

import com.unboundid.asn1.ASN1Buffer;
import com.unboundid.ldap.protocol.LDAPMessage;
import com.unboundid.ldap.sdk.ConnectThread;
import com.unboundid.ldap.sdk.DisconnectHandler;
import com.unboundid.ldap.sdk.DisconnectType;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionReader;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPMessages;
import com.unboundid.ldap.sdk.LDAPRuntimeException;
import com.unboundid.ldap.sdk.ResponseAcceptor;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Debug;
import com.unboundid.util.DebugType;
import com.unboundid.util.InternalUseOnly;
import com.unboundid.util.StaticUtils;
import java.io.IOException;
import java.io.OutputStream;
import java.net.Socket;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import javax.net.SocketFactory;
import javax.net.ssl.SSLContext;

@InternalUseOnly
final class LDAPConnectionInternals {
    private final AtomicInteger nextMessageID;
    private volatile boolean disconnectHandlerNotified;
    private final boolean synchronousMode;
    private final int port;
    private final long connectTime;
    private final LDAPConnection connection;
    private final LDAPConnectionReader connectionReader;
    private volatile OutputStream outputStream;
    private final Socket socket;
    private final String host;
    private static final ThreadLocal<ASN1Buffer> asn1Buffers = new ThreadLocal();

    LDAPConnectionInternals(LDAPConnection connection, LDAPConnectionOptions options, SocketFactory socketFactory, String host, int port, int timeout) throws IOException {
        this.connection = connection;
        this.host = host;
        this.port = port;
        if (options.captureConnectStackTrace()) {
            connection.setConnectStackTrace(Thread.currentThread().getStackTrace());
        }
        this.connectTime = System.currentTimeMillis();
        this.nextMessageID = new AtomicInteger(0);
        this.disconnectHandlerNotified = false;
        this.synchronousMode = options.useSynchronousMode();
        try {
            ConnectThread connectThread = new ConnectThread(socketFactory, host, port);
            connectThread.start();
            this.socket = connectThread.getConnectedSocket(timeout);
        }
        catch (LDAPException le) {
            Debug.debugException(le);
            throw new IOException(le.getMessage());
        }
        if (options.getReceiveBufferSize() > 0) {
            this.socket.setReceiveBufferSize(options.getReceiveBufferSize());
        }
        if (options.getSendBufferSize() > 0) {
            this.socket.setSendBufferSize(options.getSendBufferSize());
        }
        try {
            Debug.debugConnect(host, port, connection);
            this.socket.setKeepAlive(options.useKeepAlive());
            this.socket.setReuseAddress(options.useReuseAddress());
            this.socket.setSoLinger(options.useLinger(), options.getLingerTimeoutSeconds());
            this.socket.setTcpNoDelay(options.useTCPNoDelay());
            this.outputStream = this.socket.getOutputStream();
            this.connectionReader = new LDAPConnectionReader(connection, this);
        }
        catch (IOException ioe) {
            Debug.debugException(ioe);
            try {
                this.socket.close();
            }
            catch (Exception e) {
                Debug.debugException(e);
            }
            throw ioe;
        }
    }

    void startConnectionReader() {
        if (!this.synchronousMode) {
            this.connectionReader.start();
        }
    }

    LDAPConnection getConnection() {
        return this.connection;
    }

    LDAPConnectionReader getConnectionReader() {
        return this.connectionReader;
    }

    String getHost() {
        return this.host;
    }

    int getPort() {
        return this.port;
    }

    Socket getSocket() {
        return this.socket;
    }

    OutputStream getOutputStream() {
        return this.outputStream;
    }

    boolean isConnected() {
        return this.socket.isConnected();
    }

    boolean synchronousMode() {
        return this.synchronousMode;
    }

    void convertToTLS(SSLContext sslContext) throws LDAPException {
        this.outputStream = this.connectionReader.doStartTLS(sslContext);
    }

    int nextMessageID() {
        int msgID = this.nextMessageID.incrementAndGet();
        if (msgID > 0) {
            return msgID;
        }
        do {
            if (!this.nextMessageID.compareAndSet(msgID, 1)) continue;
            return 1;
        } while ((msgID = this.nextMessageID.incrementAndGet()) <= 0);
        return msgID;
    }

    void registerResponseAcceptor(int messageID, ResponseAcceptor responseAcceptor) throws LDAPException {
        if (!this.isConnected()) {
            LDAPConnectionOptions connectionOptions = this.connection.getConnectionOptions();
            boolean closeRequested = this.connection.closeRequested();
            if (connectionOptions.autoReconnect() && !closeRequested) {
                this.connection.reconnect();
                this.connection.registerResponseAcceptor(messageID, responseAcceptor);
            } else {
                throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_NOT_ESTABLISHED.get());
            }
        }
        this.connectionReader.registerResponseAcceptor(messageID, responseAcceptor);
    }

    void deregisterResponseAcceptor(int messageID) {
        this.connectionReader.deregisterResponseAcceptor(messageID);
    }

    void sendMessage(LDAPMessage message) throws LDAPException {
        if (!this.isConnected()) {
            throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_NOT_ESTABLISHED.get());
        }
        ASN1Buffer buffer = asn1Buffers.get();
        if (buffer == null) {
            buffer = new ASN1Buffer();
            asn1Buffers.set(buffer);
        }
        buffer.clear();
        try {
            message.writeTo(buffer);
        }
        catch (LDAPRuntimeException lre) {
            Debug.debugException(lre);
            lre.throwLDAPException();
        }
        try {
            OutputStream os = this.outputStream;
            buffer.writeTo(os);
            os.flush();
        }
        catch (IOException ioe) {
            Debug.debugException(ioe);
            if (message.getProtocolOpType() == 66) {
                return;
            }
            LDAPConnectionOptions connectionOptions = this.connection.getConnectionOptions();
            boolean closeRequested = this.connection.closeRequested();
            if (connectionOptions.autoReconnect() && !closeRequested) {
                this.connection.reconnect();
            }
            throw new LDAPException(ResultCode.SERVER_DOWN, LDAPMessages.ERR_CONN_SEND_ERROR.get(this.host + ':' + this.port, StaticUtils.getExceptionMessage(ioe)), ioe);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new LDAPException(ResultCode.LOCAL_ERROR, LDAPMessages.ERR_CONN_ENCODE_ERROR.get(this.host + ':' + this.port, StaticUtils.getExceptionMessage(e)), e);
        }
        finally {
            if (buffer.zeroBufferOnClear()) {
                buffer.clear();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close() {
        LDAPConnectionOptions connectionOptions;
        DisconnectHandler disconnectHandler;
        DisconnectType disconnectType = this.connection.getDisconnectType();
        String disconnectMessage = this.connection.getDisconnectMessage();
        Throwable disconnectCause = this.connection.getDisconnectCause();
        boolean closedByFinalizer = disconnectType == DisconnectType.CLOSED_BY_FINALIZER && this.socket.isConnected();
        try {
            this.connectionReader.close(false);
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        try {
            this.outputStream.close();
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        try {
            this.socket.close();
        }
        catch (Exception e) {
            Debug.debugException(e);
        }
        if (disconnectType == null) {
            if (Debug.debugEnabled(DebugType.LDAP)) {
                Debug.debug(Level.WARNING, DebugType.LDAP, "No disconnect type set for connection closed with stack trace " + StaticUtils.getStackTrace(Thread.currentThread().getStackTrace()));
            }
            disconnectType = DisconnectType.UNKNOWN;
        }
        Debug.debugDisconnect(this.host, this.port, this.connection, disconnectType, disconnectMessage, disconnectCause);
        if (closedByFinalizer && Debug.debugEnabled(DebugType.LDAP)) {
            Debug.debug(Level.WARNING, DebugType.LDAP, "Connection closed by LDAP SDK finalizer:  " + this.toString());
        }
        if ((disconnectHandler = (connectionOptions = this.connection.getConnectionOptions()).getDisconnectHandler()) != null && !this.disconnectHandlerNotified) {
            connectionOptions.setDisconnectHandler(null);
            try {
                this.disconnectHandlerNotified = true;
                disconnectHandler.handleDisconnect(this.connection, this.host, this.port, disconnectType, disconnectMessage, disconnectCause);
            }
            catch (Exception e) {
                Debug.debugException(e);
            }
            finally {
                if (connectionOptions.getDisconnectHandler() == null) {
                    connectionOptions.setDisconnectHandler(disconnectHandler);
                }
            }
            this.connection.setDisconnectInfo(disconnectType, disconnectMessage, disconnectCause);
        }
    }

    public long getConnectTime() {
        if (this.isConnected()) {
            return this.connectTime;
        }
        return -1L;
    }

    public String toString() {
        StringBuilder buffer = new StringBuilder();
        this.toString(buffer);
        return buffer.toString();
    }

    public void toString(StringBuilder buffer) {
        buffer.append("LDAPConnectionInternals(host='");
        buffer.append(this.host);
        buffer.append("', port=");
        buffer.append(this.port);
        buffer.append(", connected=");
        buffer.append(this.socket.isConnected());
        buffer.append(", nextMessageID=");
        buffer.append(this.nextMessageID.get());
        buffer.append(')');
    }
}

