package org.planx.xpath.object;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import org.planx.xpath.XPathException;

/**
 * A node set returned by an XPath expression.
 * Note: This class is not synchronized.
 * The terms "set" and "list" are used interchangably in this
 * class, since semantically it should be a set but this is not
 * inforced due to performance considerations. In fact it is a
 * list. However, the XPath implementation garuantees that only
 * <code>XNodeSet</code>s without duplicates are returned.
 **/
public class XNodeSet extends XObject implements List {
    private List list;

    /**
     * Creates an empty node set.
     **/
    public XNodeSet() {
        list = new ArrayList();
    }

    /**
     * Creates an empty node set with the specified initial capacity.
     **/
    public XNodeSet(int initialCapacity) {
        list = new ArrayList(initialCapacity);
    }

    /**
     * Creates a node set containing all elements in the specified
     * collection.
     **/
    public XNodeSet(Collection c) {
        list = new ArrayList(c);
    }

    // Conversion

    /**
     * Returns <code>true</code> if this node set
     * contains any elements. Otherwise returns <code>false</code>.
     **/
    public boolean booleanValue() {
        return list.size() > 0;
    }

    /**
     * Returns a comma-separated list in square brackets
     * of the contents of this node set. Each element in
     * the list is converted to a string using the
     * <code>toString()</code> method.
     **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("[");
        Iterator it = list.iterator();
        while (it.hasNext()) {
            sb.append(it.next().toString());
            if (it.hasNext()) sb.append(",");
        }
        sb.append("]");
        return sb.toString();
    }

    // Delegate methods to list

    /**
     * Returns the number of elements in this set.
     */
    public int size() {
        return list.size();
    }

    /**
     * Returns <tt>true</tt> if this set contains no elements.
     */
    public boolean isEmpty() {
        return list.isEmpty();
    }

    /**
     * Returns <tt>true</tt> if this set contains the specified element.
     */
    public boolean contains(Object o) {
        return list.contains(o);
    }

    /**
     * Returns an iterator over the elements in this set.
     */
    public Iterator iterator() {
        return list.iterator();
    }

    /**
     * Returns an array containing all of the elements in this set.
     */
    public Object[] toArray() {
        return list.toArray();
    }

    /**
     * Returns an array containing all of the elements in this set; the
     * runtime type of the returned array is that of the specified array.
     */
    public Object[] toArray(Object a[]) {
        return list.toArray(a);
    }

    /**
     * Appends the specified element.
     */
    public boolean add(Object o) {
        return list.add(o);
    }

    /**
     * Removes the specified element from this set if it is present.
     */
    public boolean remove(Object o) {
        return list.remove(o);
    }

    /**
     * Returns <tt>true</tt> if this set contains all of the elements of the
     * specified collection.
     */
    public boolean containsAll(Collection c) {
        return list.containsAll(c);
    }

    /**
     * Appends all of the elements in the specified collection.
     */
    public boolean addAll(Collection c) {
        return list.addAll(c);
    }

    /**
     * Inserts all of the elements in the specified collection
     * at the specified position.
     */
    public boolean addAll(int index, Collection c) {
        return list.addAll(index, c);
    }

    /**
     * Retains only the elements in this set that are contained in the
     * specified collection.
     */
    public boolean retainAll(Collection c) {
        return list.retainAll(c);
    }

    /**
     * Removes from this set all of its elements that are contained in the
     * specified collection.
     */
    public boolean removeAll(Collection c) {
        return list.removeAll(c);
    }

    /**
     * Removes all of the elements from this set.
     */
    public void clear() {
        list.clear();
    }

    /**
     * Compares the specified object with this set for equality.
     */
    public boolean equals(Object o) {
        return list.equals(o);
    }

    /**
     *
     * Returns the hash code value for this set.
     */
    public int hashCode() {
        return list.hashCode();
    }

    /**
     * Returns the element at the specified position in this list.
     */
    public Object get(int index) {
        return list.get(index);
    }

    /**
     * Replaces the element at the specified position in this list with the
     * specified element.
     */
    public Object set(int index, Object element) {
        return list.set(index, element);
    }

    /**
     * Inserts the specified element at the specified position in this list.
     */
    public void add(int index, Object element) {
        list.add(index, element);
    }

    /**
     * Removes the element at the specified position in this list.
     */
    public Object remove(int index) {
        return list.remove(index);
    }

    /**
     * Returns the index in this list of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     */
    public int indexOf(Object o) {
        return list.indexOf(o);
    }

    /**
     * Returns the index in this list of the last occurrence of the specified
     * element, or -1 if this list does not contain this element.
     */
    public int lastIndexOf(Object o) {
        return list.lastIndexOf(o);
    }


    /**
     * Returns a list iterator of the elements in this list.
     */
    public ListIterator listIterator() {
        return list.listIterator();
    }

    /**
     * Returns a list iterator of the elements in this list,
     * starting at the specified position in this list.
     */
    public ListIterator listIterator(int index) {
        return list.listIterator(index);
    }

    /**
     * Returns a view of the portion of this list between the specified
     * <tt>fromIndex</tt>, inclusive, and <tt>toIndex</tt>, exclusive.
     */
     public List subList(int fromIndex, int toIndex) {
        return list.subList(fromIndex, toIndex);
     }

}
