package org.planx.xpath;

import org.planx.xmlstore.docnodes.DocAttribute;
import org.planx.xmlstore.docnodes.DocNode;

/**
 * This navigator implements a bridge between XPath
 * and XML Store.
 *
 * @author Thomas Ambus
 **/
public class XMLStoreNavigator implements Navigator {

    /**
     * Construct an XMLStoreNavigator.
     **/
    public XMLStoreNavigator() {}

    public Object getRoot(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).getRoot();
        } else if (node instanceof DocAttribute) {
            return ((DocAttribute) node).getOwner().getRoot();
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public Object getParent(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).getParent();
        } else if (node instanceof DocAttribute) {
            return ((DocAttribute) node).getOwner();
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public Object nextSibling(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).nextSibling();
        } else if (node instanceof DocAttribute){
            return null;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public Object previousSibling(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).previousSibling();
        } else if (node instanceof DocAttribute) {
            return null;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public Object getChild(Object node, int index) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).getChild(index);
        } else if (node instanceof DocAttribute){
            return null;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public int childCount(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).childCount();
        } else if (node instanceof DocAttribute){
            return 0;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public Object getAttribute(Object node, int index) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).getAttribute(index);
        } else if (node instanceof DocAttribute){
            return null;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public int attributeCount(Object node) throws XPathException {
        if (node instanceof DocNode) {
            return ((DocNode) node).attributeCount();
        } else if (node instanceof DocAttribute) {
            return 0;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    // Data

    public String getName(Object node) throws XPathException {
        if (node instanceof DocNode) {
            DocNode docNode = (DocNode) node;
            switch (docNode.getType()) {
            case DocNode.ELEMENT:
                return docNode.getNodeValue();
            case DocNode.CHARDATA:
                return null;
            default:
                throw new RuntimeException("Unsupported type: "
                                           +docNode.getType());
            }
        } else if (node instanceof DocAttribute) {
            return ((DocAttribute) node).getName();
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    public String getStringValue(Object node) throws XPathException {
        if (node instanceof DocAttribute) {
            return ((DocAttribute) node).getValue();
        } else if (node instanceof DocNode) {
            StringBuffer sb = new StringBuffer();
            appendStringValue((DocNode) node, sb);
            return sb.toString();
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }

    /**
     * Recursively append string values of children.
     * Only text nodes actually append anything.
     **/
    private void appendStringValue(DocNode node, StringBuffer sb) {
        switch (node.getType()) {
        case DocNode.ELEMENT:
            int max = node.childCount();
            for (int i = 0; i < max; i++) {
                appendStringValue(node.getChild(i), sb);
            }
            break;
        case DocNode.CHARDATA:
            sb.append(node.getNodeValue());
            break;
        default:
            throw new RuntimeException("Unsupported type: "+node.getType());
        }
    }

    public int getType(Object node) throws XPathException {
        if (node instanceof DocNode) {
            DocNode docNode = (DocNode) node;
            switch (docNode.getType()) {
            case DocNode.ELEMENT:
                return Navigator.ELEMENT;
            case DocNode.CHARDATA:
                return Navigator.TEXT;
            default:
                throw new RuntimeException("Unsupported type: "
                                           +docNode.getType());
            }
        } else if (node instanceof DocAttribute) {
            return Navigator.ATTRIBUTE;
        } else {
            throw new ClassCastException("Unknown object: "+node);
        }
    }
}
