package org.planx.xmlstore.stores;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.references.*;

/**
 * A bridge between location independent and location dependent <code>XMLStore</code>s.
 * The <code>TranslatorXMLStore</code> translates the underlying <code>XMLStore</code>s
 * <code>Reference</code>s to <code>ContentValueReference</code>s that are location
 * independent. It uses a reference server to do so. Only root nodes (i.e. ones that
 * were at any point argument to the <code>save</code> method) are registered in the
 * reference server.
 *
 * @author Thomas Ambus
 */
public class TranslatorXMLStore extends AbstractXMLStore {
    private PersistentMap<ContentValueReference,Reference> refserver;
    private LocalXMLStore localXMLStore;

    public TranslatorXMLStore(LocalXMLStore xmlstore) throws IOException {
        super(xmlstore);
        this.localXMLStore = xmlstore;
        String name = xmlstore.toString();
        Streamer<ContentValueReference> s1 = ContentValueReference.getStreamer();
        Streamer<Reference> s2 = Streamers.getPolymorphicStreamer(Reference.class);
        refserver = new PersistentMap<ContentValueReference,Reference>
                                        (name + ".global.map", s1, s2);
        localXMLStore.addReferenceListener(new TranslatorListener());
    }

    /**
     * <code>ContentValueReference</code>s will be resolved by looking up the
     * root reference in the local reference server, loading this node in the
     * underlying store, and traversing from this node to the node designated
     * be the path of the global reference. Other references will be handed
     * directly to the underlying store, which may or may not be able to
     * translate them. <code>RelativeReference</code>s containing
     * <code>ContentValueReference</code>s as roots are accepted.
     */
    protected Node resolvedLoad(Reference vref) throws IOException,
                                         UnknownReferenceException {
        checkClosed();
        if (vref instanceof ContentValueReference) {
            Reference loc = refserver.get((ContentValueReference) vref);
            if (loc == null) throw new UnknownReferenceException(vref);
            Node root = xmlstore.load(loc);
            return root;
        } else {
            // Unknown Reference type, attempt to load from underlying store anyway
            // as it might know this type. TODO: Should this feature be removed?
            return xmlstore.load(vref);
        }
    }

    /**
     * Saves the <code>Node</code>, gives it a <code>ContentValueReference</code>,
     * and registers it in the local reference server.
     */
    public Reference save(Node node) throws IOException {
        checkClosed();
        Reference vref = xmlstore.save(node);
        ContentValueReference gref = new ContentValueReference(node);
        refserver.put(gref, vref);
        return gref;
    }

    public void retain(Reference ref) throws UnknownReferenceException {
        Reference loc = refserver.get((ContentValueReference) ref);
        localXMLStore.retain(loc);
    }

    public void release(Reference ref) throws UnknownReferenceException {
        Reference loc = refserver.get((ContentValueReference) ref);
        localXMLStore.release(loc);
    }

    public void close() throws IOException {
        super.close();
        refserver = null;
    }

    /**
     * When a reference is updated, replaces all occurrences of the old reference
     * with the new in refserver.
     */
    private class TranslatorListener extends AbstractReferenceListener {
        public void referenceMoved(Locator oldLoc, Locator newLoc) {
            try {
                refserver.replace(oldLoc, newLoc);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
