package org.planx.xmlstore.stores;

import org.planx.xmlstore.*;

import java.net.*;
import java.io.*;
import org.planx.xmlstore.util.Concurrency;

/**
 * A NetworkSkeleton exposes the functionality of an underlying
 * {@link XMLStore} to networked clients. Once created, the Stub
 * handles requests on the specified port from clients.
 * <br><br>
 * For now the Stub only understands LOAD and SAVE requests.
 * A LOAD request includes a value reference which is then
 * loaded from the underlying {@link XMLStore}. The resulting {@link Node}
 * is marshalled and sent in a response to the client.
 * <br><br>
 * A SAVE request includes either character data, or
 * element data in the form of a tag and a list of child
 * value references. A {@link Node} is created based on this
 * and then saved in the underlying {@link XMLStore}. The resulting
 * value reference is sent back to the client.
 * <br><br>
 * NetworkSkeleton is not quite a decorator as it does not implement
 * the XML Store. It is not quite an Adaptor either, as it does
 * not implement any interface.
 *
 * @author Henning Niss
 */
public class NetworkSkeleton {
    private XMLStore xmlstore;
    private ServerSocket serverSocket;
    private boolean isRunning = true;

    /**
     * Create a Network Stub XML Store which listens on the
     * specified port and handles requests using the specified
     * underlying XML Store. The store will create a default number of
     * threads to handle requests (currently 100).
     *
     * @param xmlstore   the underlying XML Store
     * @param port       the port on which to listen
     **/
    public NetworkSkeleton(XMLStore xmlstore, int port) {
        this.xmlstore = xmlstore;
        String host;
        try {
            host = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            host = "127.0.0.1";
        }

        System.out.println("Starting NetworkSkeleton on "+host+":"+port);

        try {
            serverSocket = new ServerSocket(port);
            new Thread() {
                public void run() {
                    listen();
                }
            }.start();
        } catch (IOException e) {
            System.out.println("Couldn't bind socket: " + e);
        }
    }

    private void listen() {
        try {
            while (isRunning) {
                Socket socket = serverSocket.accept();
                socket.setKeepAlive(true);
                try {
                    Runnable runnable = new NetworkRequestHandler(xmlstore, socket);
                    Concurrency.networkExecutor().execute(runnable);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } catch (SocketException e) {
            // socket closed
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public synchronized void close() throws IOException {
        // Method is synchronized since it could be called concurrently from listen()
        if (isRunning) {
            isRunning = false;
            System.out.println("Stopping NetworkSkeleton");
            serverSocket.close(); // will cause exception that breaks listen() loop
        }
    }
}
