package org.planx.xmlstore.nodes;

import org.planx.msd.lang.*;
import org.planx.util.*;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.io.LocalLocator;
import org.planx.xmlstore.io.Streamers;
import org.planx.xmlstore.references.*;
import java.util.Arrays;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;

/**
 * Standard implementation of an immutable <code>Node</code>.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class DVMNode extends AbstractNode {
    private static final List<SystemNode> EMPTY_CHILDREN   = Collections.EMPTY_LIST;
    private static final List<Attribute> EMPTY_ATTRIBUTES = Collections.EMPTY_LIST;

    private byte type;                      // XML node type: ELEMENT/CHARDATA
    private String value;                   // XML node value
    private List<SystemNode> children;      // ensured immutable
    private List<Attribute> attributes;     // ensured immutable
    private int hash = 0;                   // cache hash code
    private boolean isShared = false;       // heuristic used to cache nodes during loading
    private int height = -1;                // cache height during discrimination
    private LocalLocator loc = null;

    public DVMNode(byte type, String value) {
        if (value == null)
            throw new IllegalArgumentException("Value may not be null");
        if (type == ELEMENT && value.length() == 0)
            throw new IllegalArgumentException("ELEMENT cannot have empty value");
        this.type = type;
        this.value = value;
        this.children = EMPTY_CHILDREN;
        this.attributes = EMPTY_ATTRIBUTES;
    }

    public DVMNode(byte type, String value, List<SystemNode> children,
                                                List<Attribute> attrs) {
        if (value == null)
            throw new IllegalArgumentException("Value may not be null");

        switch (type) {
        case CHARDATA:
            if (children != null)
                throw new IllegalArgumentException
                    ("CHARDATA cannot have children");
            if (attrs != null)
                throw new IllegalArgumentException
                    ("CHARDATA cannot have attributes");
            this.children = EMPTY_CHILDREN;
            this.attributes = EMPTY_ATTRIBUTES;
            break;

        case ELEMENT:
            if (value.length() == 0)
                throw new IllegalArgumentException("ELEMENT cannot have empty value");
            this.children = (children==null || children.size()==0) ? EMPTY_CHILDREN :
                                                    Array.unmodifiableCopy(children);
            this.attributes = (attrs==null || attrs.size()==0) ? EMPTY_ATTRIBUTES :
                                                     Array.unmodifiableCopy(attrs);
            break;

        default:
            throw new IllegalArgumentException("Invalid node type");
        }

        this.type = type;
        this.value = value;
    }

    public DVMNode(byte type, String value, SystemNode[] children, Attribute[] attrs) {
        if (value == null)
            throw new IllegalArgumentException("Value may not be null");

        switch (type) {
        case CHARDATA:
            if (children != null)
                throw new IllegalArgumentException("CHARDATA cannot have children");
            if (attrs != null)
                throw new IllegalArgumentException("CHARDATA cannot have attributes");
            this.children = EMPTY_CHILDREN;
            this.attributes = EMPTY_ATTRIBUTES;
            break;

        case ELEMENT:
            if (value.length() == 0)
                throw new IllegalArgumentException("ELEMENT cannot have empty value");
            this.children = (children==null || children.length==0) ? EMPTY_CHILDREN :
                                                    Array.unmodifiableCopy(children);
            this.attributes = (attrs==null || attrs.length==0) ? EMPTY_ATTRIBUTES :
                                                     Array.unmodifiableCopy(attrs);
            break;

        default:
            throw new IllegalArgumentException("Invalid node type");
        }

        this.type = type;
        this.value = value;
    }

    /**
     * Package private constructor with no checks and no copying.
     */
    DVMNode(byte type, String value, List<SystemNode> children, List<Attribute> attrs,
                  LocalLocator loc, boolean isShared) {
        this.type = type;
        this.value = value;
        this.children = (children==null) ? EMPTY_CHILDREN : children;
        this.attributes = (attrs==null) ? EMPTY_ATTRIBUTES : attrs;
        this.loc = loc;
        this.isShared = isShared;
    }

    public boolean isMutable() {
        return false;
    }

    public byte getType() {
        return type;
    }

    public String getNodeValue() {
        return value;
    }

    public List<SystemNode> getChildren() {
        return children;
    }

    public List<Attribute> getAttributes() {
        return attributes;
    }

    public String getAttribute(String attrName) {
        for (Attribute attr : attributes)
            if (attr.getName().equals(attrName))
                return attr.getValue();
        return null;
    }

    public String[] getAttributeNames() {
        String[] names = new String[attributes.size()];
        int i=0;
        for (Attribute attr : attributes) {
            names[i++] = attr.getName();
        }
        return names;
    }

    public boolean equals(Object o) {
        if (o == this) return true;
        if (!(o instanceof Node)) return false;
        return deepEquals(this, (Node) o);
    }

    private static boolean deepEquals(Node n1, Node n2) {
        if (n1.getType() != n2.getType()) return false;
        if (!n1.getNodeValue().equals(n2.getNodeValue())) return false;

        List<Attribute> as1 = n1.getAttributes();
        List<Attribute> as2 = n2.getAttributes();
        int asize = as1.size();
        if (asize != as2.size()) return false;
        Attribute[] attrs1 = new Attribute[asize];
        attrs1 = as1.toArray(attrs1);
        Arrays.sort(attrs1);
        Attribute[] attrs2 = new Attribute[asize];
        attrs2 = as2.toArray(attrs2);
        Arrays.sort(attrs2);
        if (!Arrays.equals(attrs1, attrs2)) return false;

        List<? extends Node> cs1 = n1.getChildren();
        List<? extends Node> cs2 = n2.getChildren();
        int csize = cs1.size();
        if (csize != cs2.size()) return false;
        for (int i=0; i<csize; i++) {
            if (!deepEquals(cs1.get(i), cs2.get(i))) return false;
        }
        return true;
    }

    public int hashCode() {
        if (hash == 0) {
            int h = type;
            h = h*31 + value.hashCode();
            h = h*31 + attributes.hashCode();
            for (Node n : children) {
                h = h*31 + n.hashCode();
            }
            hash = h;
        }
        return hash;
    }

    public String toString() {
        StringBuilder result = new StringBuilder();

        if (type == Node.ELEMENT) {
            result.append("<" + getNodeValue());

            for (Attribute attr : attributes) {
                result.append(" "+attr.getName()+"=\""+attr.getValue()+"\"");
            }

            if (children.size() > 0) {
                result.append(">\n");
                for (Node node : children) {
                    result.append(node.toString() + "\n");
                }
                result.append("</" + getNodeValue() + ">");
            } else {
                result.append("/>");
            }
        } else if (type == Node.CHARDATA) {
            result.append(value);
        }
        return result.toString();
    }

    // Specified in SystemNode

    public SystemNode get() {
        return this;
    }

    public boolean isShared() {
        return isShared;
    }

    public void setShared(boolean isShared) {
        this.isShared = isShared;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }

    public void setChild(int index, SystemNode child) {
        if (!(children instanceof UnmodifiableArrayList)) {
            children = new UnmodifiableArrayList<SystemNode>(children);
        }
        ((UnmodifiableArrayList) children).internal()[index] = child;
    }

    public LocalLocator getLocator() {
        return loc;
    }

    public void setLocator(LocalLocator loc) {
        this.loc = loc;
    }
}
