package org.planx.util;

import java.util.LinkedList;

/**
 * @author Henning Niss
 */
public class ThreadPool {
    private Worker workers[];
    private LinkedList worklist;
    private Done done = new Done();

    /**
     * Create a thread pool with the default number of worker
     * threads (currently 100).
     **/
    public ThreadPool() {
        this(100);
    }

    /**
     * Create a thread pool with the specified number of worker
     * threads.
     *
     * @param size   thread pool size
     **/
    public ThreadPool(int size) {
        workers = new Worker[size];
        worklist = new LinkedList();
        for (int i=0;i<workers.length;i++) {
            workers[i] = new Worker(this, i);
            workers[i].start();
        }
    }

    /**
     * Stop the thread pool.
     * <br/><br/>
     * Calling {@link #stop} waits until all worker threads are
     * idle, then interrupts each of them (after which they should all
     * stop because, once idle, a worker thread waits for more work),
     * and then returns.
     * <br/><br/>
     * TODO: consider forcibly stopping the workers; consider trying
     * to stop the workers gracefully, and if this doesn't not happen
     * within some timeout, stop them forcibly.
     **/
    public void stop() {
        done.waitUntilDone();
        for (int i=0;i<workers.length;i++) {
            workers[i].interrupt();
        }
    }


    /**
     * Add work to the thread pool.
     *
     * @param work   work to be executed by the next available worker.
     */
    public synchronized void addWork(Runnable work) {
        worklist.add(work);
        notify();
    }

    private synchronized Runnable getWork() {
        try {
            while (worklist.isEmpty()) {
                wait();
            }
            Runnable work = (Runnable) worklist.removeFirst();
            return work;
        } catch (InterruptedException e) {
            return null;
        }
    }

    private void active(int id) {
        done.active();
    }

    private void inactive(int id) {
        done.inactive();
    }

    /**
     * Check to see if the thread pool is idle (no worker threads
     * doing useful work).
     *
     * @return false if at least one worker thread is doing useful
     *   work, true otherwise.
     */
    public boolean idle() {
        return done.idle();
    }

    private class Done {
        private int activeworkers;
        private synchronized void active() {
            activeworkers ++ ;
            notify();
        }

        private synchronized void inactive() {
            if(activeworkers>0) activeworkers --;
            notify();
        }

        public synchronized boolean idle() {
            return activeworkers == 0;
        }

        public synchronized void waitUntilDone() {
            try {
                while(activeworkers > 0) wait();
            } catch (InterruptedException e) {
                // FIXME: We should do something, I suppose
            }
        }
    }

    private class Worker extends Thread {
        ThreadPool coord;
        int id;
        private Worker(ThreadPool coord, int id) {
            this.coord = coord;
            this.id = id;
        }
        public void run() {
            Runnable work = null;
            do {
                work = coord.getWork();
                coord.active(id);
                if(work != null) {
                    work.run();
                }
                coord.inactive(id);
            } while (work != null);
        }
    }
}
