package org.planx.msd.util;

import java.util.Collection;
import java.util.ArrayList;
import java.util.List;
import org.planx.msd.*;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * values into an equivalence class containing <code>null</code> elements
 * and one or more equivalence classes containing the non-<code>null</code>
 * elements as discriminated by another discriminator. The intention is to
 * be able to extend discriminators that do not allow <code>null</code> input
 * values to allow such values.
 *
 * @author Thomas Ambus
 */
public class NullDiscriminator<T> extends AbstractDiscriminator<T> {
    private final Discriminator<T> disc;

    /**
     * Constructs a new <code>NullDiscriminator</code> wrapping the specified
     * discriminator, thereby extending the wrapped discriminator to allow
     * <code>null</code> values.
     */
    public NullDiscriminator(Discriminator<T> d) {
        disc = d;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                              Extractor<U,? extends T,S> e) {
        List<S> nulls = new ArrayList<S>();
        List<U> nonNulls = new ArrayList<U>(values.size());

        for (U elm : values) {
            if (e.getLabel(elm) == null) nulls.add(e.getValue(elm));
            else nonNulls.add(elm);
        }

        Collection<List<S>> result = disc.discriminate(nonNulls, e);

        // Add null equivalence class to result produced by wrapped
        // discriminator

        if (!nulls.isEmpty()) {
            try {
                result.add(nulls);
            } catch (UnsupportedOperationException uoe) {
                result = new ArrayList<List<S>>(result);
                result.add(nulls);
            }
        }
        return result;
    }
}
