package org.planx.xmlstore.stores;

import java.io.IOException;
import org.planx.xmlstore.*;
import org.planx.xmlstore.references.*;

/**
 * A convenience class for implementing <code>XMLStore</code>s. The implementation
 * of the {@link #load} method resolves any {@link RelativeReference}s before calling
 * {@link #resolvedLoad}. This method calls the underlying <code>XMLStore</code> if
 * one was provided in the constructor.
 * Thus, implementations that only accept certain types of <code>Reference</code>s
 * should override the {@link #resolvedLoad} method instead of the {@link #load} method.
 * Implementations that accept any kind of <code>Reference</code> can just
 * override the {@link #load} method.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public abstract class AbstractXMLStore implements XMLStore {
    protected XMLStore xmlstore;

    /**
     * Base <code>XMLStore</code>s that do not wrap other <code>XMLStore</code>s
     * can use this constructor.
     */
    protected AbstractXMLStore() {}

    /**
     * Decorators that wrap other <code>XMLStore</code>s can use this
     * constructor.
     */
    protected AbstractXMLStore(XMLStore xmlstore) {
        this.xmlstore = xmlstore;
    }

    public void close() throws IOException {
        xmlstore.close();
        xmlstore = null;
    }

    protected void checkClosed() {
        if (xmlstore == null) throw new IllegalStateException("XMLStore is closed");
    }

    public Reference save(Node node) throws IOException {
        checkClosed();
        return xmlstore.save(node);
    }

    /**
     * Resolves any {@link RelativeReference}s and calls {@link #resolvedLoad}.
     */
    public Node load(Reference vref) throws IOException,
                              UnknownReferenceException {
        checkClosed();
        return loadResolve(vref);
    }

    private Node loadResolve(Reference vref) throws IOException, UnknownReferenceException {
        if (vref instanceof RelativeReference) {
            RelativeReference relRef = (RelativeReference) vref;
            Node root = load(relRef.getRoot());
            return findNode(root, relRef.getPath(), 0);
        } else if (vref instanceof ReferenceProxy) {
            return loadResolve(((ReferenceProxy) vref).get());
        } else {
            return resolvedLoad(vref);
        }
    }

    /**
     * Called by {@link #load} after resolving {@link RelativeReference}s.
     *
     * @param resolvedRef  the resolved <code>Reference</code>
     */
    protected Node resolvedLoad(Reference resolvedRef) throws IOException,
                                                UnknownReferenceException {
        return xmlstore.load(resolvedRef);
    }

    private static Node findNode(Node node, int[] path, int pos) {
        if (pos >= path.length) return node;
        else return findNode(node.getChildren().get(path[pos]), path, pos+1);
    }

    public String toString() {
        return xmlstore.toString();
    }
}
