package org.planx.xmlstore.routing;

import java.io.IOException;
import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.List;

/**
 * Maps keys to values in a distributed setting using <code>Identifier</code>s
 * as keys. Methods throw IOException if a network error occurs and
 * RoutingException if a routing specific error occurs in the overlay network.
 * Note that RoutingException is a subclass of IOException so if the
 * distinction is not needed, IOException is the only checked exception that
 * needs to be handled.
 *
 * @author Thomas Ambus
 **/
public interface DistributedMap {
    /**
     * Closes the map and releases all ressources.
     * Any subsequent calls to methods are invalid.
     **/
    public void close() throws IOException;

    /**
     * Returns <code>true</code> if the map contains the specified key and
     * <code>false</code> otherwise.
     **/
    public boolean contains(Identifier key) throws IOException, RoutingException;

    /**
     * Returns the value associated with the specified key or <code>null</code>
     * if does not exist.
     **/
    public Serializable get(Identifier key) throws IOException, RoutingException;

    /**
     * Associates the specified value with the specified key.
     **/
    public void put(Identifier key, Serializable value) throws IOException, RoutingException;

    /**
     * Removes the mapping with the specified key from this map (optional operation).
     * If the underlying routing algorithm does not support removing mappings this
     * method should do nothing.
     **/
    public void remove(Identifier key) throws IOException, RoutingException;

    /**
     * Returns the contact port set using the {@link #setContactPort} method.
     **/
    public int getContactPort();

    /**
     * Sets the contact port of this peer. This can be either a TCP or an UDP port.
     * The interpretation is entirely up to the application programmer. The contact
     * port is retrieved by other peers invoking the {@link #responsiblePeers} method.
     * The behavoir of setting the contact port after bootstrapping is unspecified.
     **/
    public void setContactPort(int port);

    /**
     * Returns a List of {@link InetSocketAddress}es identifying the peers
     * responsible for the specified key according to the underlying
     * routing algorithm. The InetSocketAddresses consist of the IP number of
     * the peer and the contact port specified using the {@link #setContactPort}
     * method at the peer.
     * Note that the number of returned peers is not specified - the specific
     * number will be implementation dependent.
     **/
    public List responsiblePeers(Identifier key) throws IOException, RoutingException;

    /**
     * Adds a <code>NeighbourhoodListener</code>.
     */
    public void addNeighbourhoodListener(NeighbourhoodListener listener);

    /**
     * Removes a <code>NeighbourhoodListener</code>.
     */
    public void removeNeighbourhoodListener(NeighbourhoodListener listener);
}
