package org.planx.xmlstore.nodes;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.planx.util.Array;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.io.Streamers;

/**
 * A <code>Streamer</code> of <code>Node</code>s that does not use references,
 * but inlines each child node directly in the output. Thus, shared subtrees
 * will be duplicated in the output. Note, that <code>NodeProxy</code>s will
 * be resolved before writing them to the output.
 *
 * @author Thomas Ambus
 */
public class InlineNodeStreamer implements Streamer<Node> {
    private static Streamer<Attribute> attrStreamer = DVMAttribute.getStreamer();

    public void toStream(DataOutput out, Node node) throws IOException {
        byte type = node.getType();
        out.writeByte(type);
        Streamers.writeUTF(out, node.getNodeValue());
        if (type == Node.ELEMENT) {
            Streamers.writeList(out, node.getAttributes(), attrStreamer);
            List<? extends Node> children = node.getChildren();
            Streamers.writeShortInt(out, children.size());
            for (Node n : children) {
                toStream(out, n);
            }
        }
    }

    public SystemNode fromStream(DataInput in) throws IOException {
        byte type = in.readByte();
        String value = Streamers.readUTF(in);
        List<Attribute> attributes = null;
        List<SystemNode> children = null;

        if (type == Node.ELEMENT) {
            attributes = (List) Streamers.readList(in, attrStreamer);
            int size = Streamers.readShortInt(in);
            children = new ArrayList<SystemNode>(size);
            for (int i=0; i<size; i++) {
                SystemNode n = fromStream(in);
                children.add(n);
            }
        }
        // use unchecked constructor
        return new DVMNode(type, value, children, attributes, null, false);
    }
}
