package org.planx.xmlstore.nodes;

import java.util.Collections;
import java.util.List;
import org.planx.util.*;
import org.planx.msd.Discriminator;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.lang.PolymorphicDiscriminator;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.references.*;

/**
 * An implementation of <code>NodeFactory</code> that operates strictly on
 * <code>SystemNode</code>s. The preferred way to obtain a <code>SystemNode</code>
 * from an unknown <code>Node</code> (which may already be a <code>SystemNode</code>)
 * is <i>not</i> to cast it but to use the {@link #convert(Node)} method provided
 * here.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public final class DVMNodeFactory extends NodeFactory {
    public DVMNodeFactory() {
        PolymorphicStreamer<Reference> ps =
            Streamers.getPolymorphicStreamer(Reference.class);

        ps.addStreamer(ContentValueReference.CLASS_ID,
                       ContentValueReference.class,
                       ContentValueReference.getStreamer());

        ps.addStreamer(RelativeReference.CLASS_ID,
                       RelativeReference.class,
                       RelativeReference.getStreamer());

        ps.addStreamer(ReferenceProxy.CLASS_ID,
                       ReferenceProxy.class,
                       ReferenceProxy.getStreamer());

        PolymorphicDiscriminator<Reference> pd =
            DiscriminatorFactory.instance().getPolymorphicDiscriminator(Reference.class);

        pd.addDiscriminator(ContentValueReference.class,
                            ContentValueReference.getDiscriminator());

        pd.addDiscriminator(RelativeReference.class,
                            RelativeReference.getDiscriminator());

        pd.addDiscriminator(ReferenceProxy.class,
                            ReferenceProxy.getDiscriminator());
    }

    public Attribute createAttribute(String name, String value) {
        return new DVMAttribute(name, value);
    }

    public Node createElementNode(String tagName) {
        return new DVMNode(Node.ELEMENT, tagName);
    }

    public Node createElementNode(String tagName, List<? extends Node> children) {
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, tagName, convert(children), null, null, false);
    }

    public Node createElementNode(String tagName, List<? extends Node> children,
                                                          List<Attribute> attrs) {
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, tagName, convert(children),
                                 Array.unmodifiableCopy(attrs), null, false);
    }

    public Node createElementNode(String tagName, Node[] children) {
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, tagName, convert(children), null, null, false);
    }

    public Node createElementNode(String tagName, Node[] children, Attribute[] attrs) {
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, tagName, convert(children),
                                 Array.unmodifiableCopy(attrs), null, false);
    }

    public Node createElementNode(String tagName, Node child) {
        List<SystemNode> children = new UnmodifiableArrayList<SystemNode>
                                     (new SystemNode[] {convert(child)});
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, tagName, children, null, null, false);
    }

    public Node createCharDataNode(String data) {
        return new DVMNode(Node.CHARDATA, data);
    }

    /**
     * Addition method that converts any <code>Node</code> to a <code>SystemNode</code>.
     * If it is already a <code>SystemNode</code> it will merely be cast.
     */
    public static SystemNode convert(Node n) {
        if (n instanceof SystemNode) return (SystemNode) n;
        if (n.getType() == Node.CHARDATA)
            return new DVMNode(Node.CHARDATA, n.getNodeValue());
        if (n.getType() != Node.ELEMENT) throw new IllegalArgumentException("Unknown node type");
        // unchecked constructor
        return new DVMNode(Node.ELEMENT, n.getNodeValue(), convert(n.getChildren()),
                                 Array.unmodifiableCopy(n.getAttributes()));
    }

    public Node replaceChild(Node node, int index, Node child) {
        if (node.getType() != Node.ELEMENT)
            throw new IllegalArgumentException("Node must be ELEMENT type");
        List<? extends Node> childNodes = node.getChildren();
        int size = childNodes.size();
        SystemNode[] children = new SystemNode[size];
        for (int i=0; i<size; i++) {
            if (i == index) children[i] = convert(child);
            else children[i] = convert(childNodes.get(i));
        }
        List<SystemNode> cs = new UnmodifiableArrayList<SystemNode>(children); // no copying
        // use unchecked constructor
        return new DVMNode(Node.ELEMENT, node.getNodeValue(), cs,
                                 node.getAttributes(), null, false);
    }

    public Node insertChild(Node n, int index, Node child) {
        List<? extends Node> childNodes = n.getChildren();
        SystemNode[] children = new SystemNode[childNodes.size()+1];

        for (int i=0; i<index; i++) {
            children[i] = convert(childNodes.get(i));
        }
        children[index] = convert(child);

        for (int i=index+1; i<children.length; i++) {
            children[i] = convert(childNodes.get(i-1));
        }
        List<SystemNode> cs = new UnmodifiableArrayList<SystemNode>(children); // no copying
        // use unchecked constructor
        return new DVMNode(Node.ELEMENT, n.getNodeValue(), cs,
                                 n.getAttributes(), null, false);
    }

    public Node removeChild(Node n, int index) {
        List<? extends Node> childNodes = n.getChildren();
        SystemNode[] children = new SystemNode[childNodes.size()-1];

        for (int i=0; i<index; i++) {
            children[i] = convert(childNodes.get(i));
        }
        for (int i=index; i<children.length; i++) {
            children[i] = convert(childNodes.get(i+1));
        }
        List<SystemNode> cs = new UnmodifiableArrayList<SystemNode>(children); // no copying
        // use unchecked constructor
        return new DVMNode(Node.ELEMENT, n.getNodeValue(), cs,
                                 n.getAttributes(), null, false);
    }

    private static List<SystemNode> convert(List<? extends Node> nodes) {
        int size = nodes.size();
        SystemNode[] sys = new SystemNode[size];
        for (int i=0; i<size; i++) {
            sys[i] = convert(nodes.get(i));
        }
        return new UnmodifiableArrayList<SystemNode>(sys); // no copying
    }

    private static List<SystemNode> convert(Node[] nodes) {
        int size = nodes.length;
        SystemNode[] sys = new SystemNode[size];
        for (int i=0; i<size; i++) {
            sys[i] = convert(nodes[i]);
        }
        return new UnmodifiableArrayList<SystemNode>(sys); // no copying
    }
}
