package org.planx.xmlstore.nodes;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Stack;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.XMLStore;
import org.planx.util.UnmodifiableArrayList;
import org.xml.sax.*;
import org.xml.sax.helpers.*;

public class DVMHandler extends DefaultHandler {
    private NodeListener listener;
    private Stack<Element> elementStack;
    private boolean hasCharData;
    private StringBuilder charBuf;
    private SystemNode root;

    public DVMHandler() {
        this(null);
    }

    public DVMHandler(NodeListener listener) {
        if (listener == null) {
            listener = new NodeListener() {
                public SystemNode nodeCreated(SystemNode node, int depth) {
                    return node;
                }
            };
        }
        this.listener = listener;
    }

    public SystemNode getRoot() {
        return root;
    }

    /**
     * Releases all ressources held by the handler. The handler should not be used again.
     * This is done to ensure that a parser with a reference to this handler does not
     * keep otherwise un-referenced data alive.
     */
    public void clear() {
        listener = null;
        elementStack = null;
        charBuf = null;
        root = null;
    }

    public void startDocument() {
        elementStack = new Stack<Element>();
        charBuf = new StringBuilder();
        hasCharData = false;
        root = null;
    }

    public void endDocument() {
        elementStack = null;
        charBuf = null;
        hasCharData = false;
    }

    public void startElement(String uri, String localName, String qName,
                                                  Attributes attributes) {
        if (qName.trim().length() > 0) {
            checkCharData();
            elementStack.push(new Element(qName, attributes));
        }
    }

    public void endElement(String uri, String localName, String qName) {
        if (qName.trim().length() == 0) return;
        checkCharData();
        Element elm = elementStack.pop();
        List<SystemNode> children = new UnmodifiableArrayList<SystemNode>(elm.children);
        SystemNode n = new DVMNode(Node.ELEMENT, elm.name, children, elm.attrs, null, false);
        SystemNode nref = listener.nodeCreated(n, elementStack.size());
        if (elementStack.empty()) {
            root = n;
        } else {
            elementStack.peek().children.add(nref);
        }
    }

    public void characters(char[] ch, int start, int length) {
        String chars = new String(ch, start, length).trim();
        if (chars.length() > 0) {
            hasCharData = true;
            charBuf.append(chars);
        }
    }

    private void checkCharData() {
        if (hasCharData) {
            String s = charBuf.toString();
            SystemNode n = new DVMNode(Node.CHARDATA, s);
            charBuf = new StringBuilder();
            hasCharData = false;
            SystemNode nref = listener.nodeCreated(n, elementStack.size());
            elementStack.peek().children.add(nref);
        }
    }

    private class Element {
        String name;
        List<Attribute> attrs;
        List<SystemNode> children = new ArrayList<SystemNode>();

        Element(String name) {
            this(name, null);
        }

        Element(String name, Attributes attributes) {
            this.name = name;
            if (attributes == null) {
                this.attrs = Collections.EMPTY_LIST;
            } else {
                Attribute[] as = new Attribute[attributes.getLength()];
                for (int i=0; i<as.length; i++) {
                    String n = attributes.getQName(i);
                    String v = attributes.getValue(i);
                    as[i] = new DVMAttribute(n,v);
                }
                this.attrs = new UnmodifiableArrayList<Attribute>(as);
            }
        }
    }
}
