package org.planx.xmlstore.io;

import java.io.*;
import java.util.*;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class PersistentMap<K,V> {
    private Map<K,V> map;
    private DataOutputStream out;
    private Streamer<K> s1;
    private Streamer<V> s2;
    private boolean isSet;

    public PersistentMap(String filename, Streamer<K> s1, Streamer<V> s2)
                                                    throws IOException {
        this(filename, s1, s2, false);
    }

    /**
     * If <code>isSet</code> is <code>true</code> the <code>Map</code>
     * will ignore values (saves space on disk).
     */
    public PersistentMap(String filename, Streamer<K> s1, Streamer<V> s2,
                           boolean isSet) throws IOException {
        this.s1 = s1;
        this.s2 = s2;
        this.isSet = isSet;
        map = new HashMap<K,V>();
        out = new DataOutputStream(
                  new BufferedOutputStream(
                      new FileOutputStream(filename, true)));
        read(filename);
    }

    private void read(String filename) throws IOException {
        DataInputStream in = new DataInputStream(
                                 new BufferedInputStream(
                                     new FileInputStream(filename)));
        try {
            // loop until EOFException
            while (true) {
                K key = s1.fromStream(in);
                V value = isSet ? null : s2.fromStream(in);
                map.put(key, value);
            }
        } catch (EOFException e) {
            // ok condition
        } finally {
            in.close();
        }
    }

    public synchronized V get(K key) {
        return map.get(key);
    }

    public synchronized void put(K key, V value) throws IOException {
        map.put(key, value);
        s1.toStream(out, key);
        if (!isSet) s2.toStream(out, value);
        out.flush();
    }

    public synchronized void replace(V oldValue, V newValue)
                            throws IOException {
        for (Map.Entry<K,V> entry : map.entrySet()) {
            if (oldValue.equals(entry.getValue())) {
                entry.setValue(newValue);
                s1.toStream(out, entry.getKey());
                if (!isSet) s2.toStream(out, newValue);
            }
        }
    }

    public synchronized Set<K> keySet() {
        return map.keySet();
    }

    public synchronized Collection<V> values() {
        return map.values();
    }

    public synchronized String toString() {
        return map.toString();
    }
}
