package org.planx.xmlstore.io;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import org.planx.msd.Discriminator;
import org.planx.msd.Memory;

/**
 * An abstraction of a <code>FileSystem</code> supporting operations needed by
 * disk <code>XMLStore</code> implementations. Every <code>FileSystem</code> has a
 * unique <code>FileSystemIdentifier</code> that can be used to identify it. All
 * <code>Locator</code>s returned by a <code>FileSystem</code> will contain this
 * <code>FileSystemIdentifier</code> so that the origin of the <code>Locator</code>
 * can be determined.
 *
 * @author Thomas Ambus
 */
public interface FileSystem {
    /**
     * Copies the data at <code>fromLoc</code> in the specified
     * <code>FileSystem</code> to <code>toLoc</code> in this
     * <code>FileSystem</code>.
     */
    public void copy(FileSystem fs, LocalLocator fromLoc, LocalLocator toLoc)
                                 throws IOException, UnknownLocatorException;

    /**
     * Positions the input at the specified location and returns a
     * <code>DataInput</code> for reading. The <code>DataInput</code> is
     * shared so if multiple threads operate on the same
     * <code>FileSystem</code> concurrently, external synchronization should
     * be used.
     */
    public DataInput getInput(LocalLocator loc) throws IOException,
                                            UnknownLocatorException;

    /**
     * Positions the output at the specified location and returns a
     * <code>DataOutpu</code> for writing. Note, that this can overwrite
     * existing data. The <code>DataInput</code> is shared so if multiple
     * threads operate on the same <code>FileSystem</code>
     * concurrently, external synchronization should be used.
     * When writing new data {@link #allocate} should be called first.
     */
    public DataOutput getOutput(LocalLocator loc) throws IOException,
                                            UnknownLocatorException;

    /**
     * Returns a <code>LocalLocator</code> to a free write position that
     * can hold <code>size</code> number of bytes.
     */
    public LocalLocator allocate() throws IOException;
    public LocalLocator allocate(int size) throws IOException;
    public LocalLocator allocate(int size, FileSystemIdentifier fsi)
                                                throws IOException;
    public FileSystemIdentifier currentIdentifier();

    /**
     * Returns <code>true</code> if the specified <code>LocalLocator</code>
     * originates from this <code>FileSystem</code>.
     */
    public boolean isContained(LocalLocator loc);

    /**
     * Returns a <code>LocalLocator</code> to all content currently in this
     * <code>FileSystem</code>.
     */
    public LocalLocator all() throws IOException;

    /**
     * Frees the space occupied by the specified <code>LocalLocator</code>.
     */
    public void free(LocalLocator loc) throws IOException,
                                    UnknownLocatorException;

    /**
     * Returns the size of the <code>FileSystem</code>.
     */
    public long size() throws IOException;

    /**
     * Closes the file system and releases all resources.
     */
    public void close() throws IOException;
}
