package org.planx.msd.character;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.RandomAccess;
import org.planx.msd.*;
import org.planx.msd.util.*;
import org.planx.util.Array;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * <code>CharSequence</code> objects. The equivalence classes are
 * returned in lexicographic order.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
public class LexicographicCharSequenceDiscriminator<T extends CharSequence>
                                extends AbstractDiscriminator<T> {
    private static final int INIT_CAPACITY = 10;
    private Memory memory;

    /**
     * Constructs a new <code>LexicographicCharSequenceDiscriminator</code>
     * reusing the memory allocated in the specified <code>Memory</code>.
     */
    public LexicographicCharSequenceDiscriminator(Memory memory) {
        this.memory = memory;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                              Extractor<U,? extends T,S> e) {
        // Check fast way out: only zero, or one element
        int vsize = values.size();
        switch (vsize) {
        case 0:
            return Collections.emptyList();
        case 1:
            List<S> l = Collections.singletonList(e.getValue(values.get(0)));
            return Collections.singletonList(l);
        }
        if (!(values instanceof RandomAccess)) values =
            new ArrayList<U>(values);

        // Declarations

        List[] dictionary = memory.dictionary;
        Collection<List<S>> result = new ArrayList<List<S>>();
        List<S> finished = new ArrayList<S>();

        int[] indexes = new int[INIT_CAPACITY]; // Note: Initialized to 0
        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 1;
        work[0] = values;                  // Start with one big block

        // Refine until there are no unfinished blocks

        while (work_head > 0) {
            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            List<U> block = work[work_head];
            int blockSize = block.size();

            if (blockSize == 1) {
                // If block only contains one element, it is finished
                result.add(Discriminators.valueList(block, e));
            } else {
                // Subpartition current unfinished block

                int initSubSize = (blockSize < 10) ? blockSize : 10;
                int index = indexes[work_head];
                int min_used = Integer.MAX_VALUE;
                int max_used = 0;

                for (int i=0; i<blockSize; i++) {
                    U elm = block.get(i);
                    CharSequence seq = e.getLabel(elm);

                    if (seq.length() <= index) {
                        // No more chars, CharSequence is finished
                        finished.add(e.getValue(elm));
                    } else {
                        int c = (int) seq.charAt(index);
                        if (c > max_used) max_used = c;
                        if (c < min_used) min_used = c;

                        List<U> list = dictionary[c];
                        if (list == null) {
                            list = new ArrayList<U>(initSubSize);
                            dictionary[c] = list;
                        }
                        list.add(elm);
                    }
                }

                // If the end of any CharSeqs was reached,
                // add these to result

                if (!finished.isEmpty()) {
                    result.add(finished);
                    finished = new ArrayList<S>();
                }

                // Add blocks in partition to result or unfinished

                index++;
                for (int idx=max_used; idx>=min_used; idx--) {
                    List<U> subBlock = dictionary[idx];
                    if (subBlock != null) {
                        dictionary[idx] = null;
                        if (work_head+1 >= work_capacity) {
                            work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                            indexes = Array.ensureCapacity(
                                indexes, work_head, work_capacity+1);
                            work_capacity = work.length;
                        }
                        work[work_head] = subBlock;
                        indexes[work_head] = index;
                        work_head++;
                    }
                }
            }
        }
        return result;
    }

    /**
     * Lexicographically sorts the specified list of
     * <code>CharSequence</code>s in-place.
     */
    public void sort(List<T> values) {

        // Check fast way out: only zero, or one element

        int vsize = values.size();
        if (vsize <= 1) return;

        // Declarations

        List[] dictionary = memory.dictionary;

        int[] indexes = new int[INIT_CAPACITY]; // Note: Initialized to 0
        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 1;
        work[0] = values;                    // Start with one big block

        int cursor = 0;

        // Refine until there are no unfinished blocks

        while (work_head > 0) {
            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            List<T> block = work[work_head];
            int blockSize = block.size();

            if (blockSize == 1) {
                // If block only contains one element, it is finished
                values.set(cursor++, block.get(0));
            } else {
                // Subpartition current unfinished block

                int initSubSize = (blockSize < 10) ? blockSize : 10;
                int index = indexes[work_head];
                int min_used = Integer.MAX_VALUE;
                int max_used = 0;

                for (int i=0; i<blockSize; i++) {
                    T seq = block.get(i);

                    if (seq.length() <= index) {
                        // No more chars, CharSequence is finished
                        values.set(cursor++, seq);
                    } else {
                        int c = (int) seq.charAt(index);
                        if (c > max_used) max_used = c;
                        if (c < min_used) min_used = c;

                        List<T> list = dictionary[c];
                        if (list == null) {
                            list = new ArrayList<T>(initSubSize);
                            dictionary[c] = list;
                        }
                        list.add(seq);
                    }
                }

                // Add blocks in partition to result or unfinished

                index++;
                for (int idx=max_used; idx>=min_used; idx--) {
                    List<T> subBlock = dictionary[idx];
                    if (subBlock != null) {
                        dictionary[idx] = null;
                        if (work_head+1 >= work_capacity) {
                            work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                            indexes = Array.ensureCapacity(
                                indexes, work_head, work_capacity+1);
                            work_capacity = work.length;
                        }
                        work[work_head] = subBlock;
                        indexes[work_head] = index;
                        work_head++;
                    }
                }
            }
        }
    }
}
